#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <pthread.h>
#include <limits.h>
#include "CANopen.h"
#include "lock.h"
#include "CanBasic.h"
#include "CanTimeStamp.h"
#include "Event.h"
#include "ComObj.h"
#include "nmt.h"
#include "ObjDict.h"

#define noNodes 127
#define noTPDO 128
#define noRPDO 128

/* 
Locking order for CANopen

   NMT0 lock                        (lowest)
   CANopen lock
   boot lock
   SDO locks (one at a time)
   SYNCer lock
   TASK locks (one at a time)
   VM memory lock                   (highest)
*/


static void initBasic() {
   initComObjEvents();
   initNmtEvents();
}   

static int firstBoot;

int initCANopen(ComponentInst *inst) {
   int iret;
   if (strcmp(inst->instName, "FtIrCANopen")) return 0;
   inst->instSpecific = NULL;
   mappedOD = (ProcessOD *)(inst->objDict);
   initBasic();
   firstBoot = 1;
   iret = pthread_attr_init(&nmtAttrs);
   iret = pthread_attr_setstacksize(&nmtAttrs, (0x20000 > PTHREAD_STACK_MIN) ? 0x20000 : PTHREAD_STACK_MIN);
   return 1;
}

#define SYNCrcvdEvent            0
#define SYNCwindowEvent          1
#define TPDOsentEvent0           2
#define TPDOsentEventN           (TPDOsentEvent0 + noTPDO - 1)
#define RPDOrcvdEvent0           (TPDOsentEventN + 1)
#define RPDOrcvdEventN           (RPDOrcvdEvent0 + noRPDO - 1)
#define TIMErcvdEvent            (RPDOrcvdEventN + 1)
#define EMCYrcvdEvent0           (TIMErcvdEvent + 1)
#define EMCYrcvdEventN           (EMCYrcvdEvent0 + noNodes - 1)
#define LocalNMTstateChange      (EMCYrcvdEventN + 1)
#define RemoteErrorControlEvent  (LocalNMTstateChange + 1)
#define LifeGuardingEvent        (RemoteErrorControlEvent + 1)
#define BootUpEvent              (LifeGuardingEvent + 1)
#define LocalErrorEvent          (BootUpEvent + 1)

#define RPDOBufferSiz            16

static vm *localErrorTask = NULL;
static vm *localNmtTask = NULL;

static void RPDOTriggerFun(vm *task) {
   int pdono = task->triggerIdx - RPDOrcvdEvent0;
   RPdoType *pdo = RPDO + pdono;
   unsigned char datalen;
   union {
      uint64_t i;
      unsigned char c[8];
   } data;
   int32_t ts;

   if (pthread_mutex_lock(&(task->mutex))) {
      fprintf(stderr, "stdTriggerFun, pthread_mutex_lock"); return;
   }
   // printf("task %s waiting\n", task->name);
   while(pdo->cnt == 0) {
      task->running = 0;
      if (pthread_cond_wait(&(task->cond), &(task->mutex))) {
         fprintf(stderr, "stdTriggerfun, pthread_cond_wait");
         pthread_mutex_unlock(&(task->mutex)); return;
      }
   }
   data.i = pdo->msg[pdo->out].data.i;
   datalen = pdo->msg[pdo->out].datalen;
   ts = pdo->msg[pdo->out].ts;
   (pdo->out)++;
   if (pdo->out == pdo->siz) pdo->out = 0;
   (pdo->cnt)--;
   pthread_mutex_unlock(&(task->mutex));
   if (!UnPack(data.c, datalen, RPdoMapping + pdono, ts))
      ReportEmcy(0x8210);
   task->statics[0] = ts;
   // printf("task %s running\n", task->name);
}

void armCANopen(vm *task) {
   int32_t eventNo = task->triggerIdx;
   ComponentInst *inst = task->triggerInst;

   if (inst->events[eventNo] != NULL) {
      task->triggerFun = stdTriggerFun;
      if (eventNo == SYNCrcvdEvent) {
         syncTask = task;
      } else if (eventNo == SYNCwindowEvent) {
         syncWindowTask = task;
      } else if (TPDOsentEvent0 <= eventNo && eventNo <= TPDOsentEventN) {
         TPDO[eventNo - TPDOsentEvent0].task = task;
      } else if (RPDOrcvdEvent0 <= eventNo && eventNo <= RPDOrcvdEventN) {
         RPDO[eventNo - RPDOrcvdEvent0].task = task;     
         RPDO[eventNo - RPDOrcvdEvent0].in = 0;     
         RPDO[eventNo - RPDOrcvdEvent0].out = 0;     
         RPDO[eventNo - RPDOrcvdEvent0].cnt = 0;
         RPDO[eventNo - RPDOrcvdEvent0].siz = RPDOBufferSiz;
         if ((RPDO[eventNo - RPDOrcvdEvent0].msg = malloc(RPDOBufferSiz * sizeof(RPDOBuffer))) == NULL) {
            fprintf(stderr, "armCANopen, cannot allocate pdo buffer");
            exit(EXIT_FAILURE);
         }  
         RPDO[eventNo - RPDOrcvdEvent0].task = task;     
         task->triggerFun = RPDOTriggerFun;
      } else if (eventNo == TIMErcvdEvent) {
         timeRcvTask = task;
      } else if (EMCYrcvdEvent0 <= eventNo && eventNo <= EMCYrcvdEventN) {
         consumedEmcy[eventNo - EMCYrcvdEvent0].task = task;
      } else if (eventNo == LocalNMTstateChange) {
         localNmtTask = task;
      } else if (eventNo == RemoteErrorControlEvent) {
         RemoteErrorControlTask = task;
      } else if (eventNo == LifeGuardingEvent) {
         LifeTimeTask = task;
      } else if (eventNo == BootUpEvent) {
         bootedTask = task;
      } else if (eventNo == LocalErrorEvent) {
         localErrorTask = task;
      }
   }
}

static void CanError(reasonTy reason) {
   // printf("in CanError, reason = %d\n", reason);
   if (reason == rxovrrn) {
      ReportEmcy(0x8110);
   } else if (reason == busoff) {
      ChkNRestartChip();
   } else if (reason == warn) {
      ReportEmcy(0x8120);
   } else ClearEmcy(0x8120);
   if (localErrorTask != NULL) {
      localErrorTask->statics[0] = reason;
      stdWakeup(localErrorTask);
   }
}

// static void LifeTimeInd(NmtStateTy foo, int occured) {
   // interface to LifeGuardingEvent
// }

void StartTimeApplication() {
   if (timeCobid & 0x80000000) {
      timeCobid &= 0x7FFFFFFF;   // time cobid is irregular, bit 31 set normally means cob invalid!!!
      StartTimeConsumer();
   } else if (timeCobid & 0x40000000) {
      StartTimeProducer();
   }
}

uint8_t *CDCF = NULL;

void readConcise(FILE *DCF) {
   uint32_t length = 1024, l = 0, c;
   CDCF = malloc(length);
   for(;;) {
      c = fread(CDCF + l, 1, length - l, DCF);
      if (c < length - l) break;
      l = length;
      length *= 2;
      CDCF = realloc(CDCF, length);
   }
   l += c;
   CDCF[l] = 0xff;
}

void NmtChngd(NmtStateTy oldst, StateCngCmdTy cmd) {
   int startall = 0;
   fprintf(stderr, "NMT state machine, old state =  %d, new state = %d\n", oldst, NmtState);
   if (NmtState == oldst) return;
   if (firstBoot && oldst != unknown) return;	// let the first bootup of a Slave finish before changing state due to NMT cmds
   // first stop what should not be running...
   if (NmtState != operational && oldst == operational) StopPDOs();
   if (NmtState == boot && (oldst == operational || oldst == preop || oldst == stopped)) {
      StopErrorControl();
   }
   if (NmtState != preop && NmtState != operational && (oldst == preop || oldst == operational)) {
      StopSync(); StopTime(); StopEmcy();
   }

   if (NmtState == boot && (cmd == resetAppl || cmd == resetApplNStp)) {
      initCanTimeStampClock();
      if (cmd == resetAppl) cmd = resetCom;
   }
   if (NmtState == boot && cmd == resetCom) {
      SetupCommParameters();
      InitObjectDictionary(NmtId, CDCF);
      InitNmt(NMTStartup & 0x1, NmtChngd);
      Bootup();
      oldst = boot; NmtState = preop; cmd = enterPreop;
   }
   if (NmtState == preop && (oldst == boot || oldst == stopped)) {
      if (oldst == boot) {
         if (HBtime != 0) {
            StartHeartBeat();
            // printf("HeartBeat started\n");
         } else {
            SetupGuarding();
         }
      }
      syncCob &= 0x7FFFFFFF;
      if (syncCob & 0x40000000) {
         StartSyncProducer();
      } else {
         StartSyncConsumer(); 
      }
      StartTimeApplication();
      SetupEmcy();
      // printf("preop init finished\n");
      if (NMTStartup & 0x1) {
         BootNetwork();
         startall = (!(NMTStartup & 0x8)) && (NMTStartup & 0x2);
      }
      if ((NMTStartup & 0x1) && !(NMTStartup & 0x4)) {
         oldst = preop; NmtState = operational;
      }
   } else if (NmtState == preop && oldst == operational) {
      if (!(NMTStartup & 0x4)) {
         oldst = preop; NmtState = operational;
      }
   } 
   if (NmtState == operational && oldst != operational) {
      if (oldst == stopped) {
         syncCob &= 0x7FFFFFFF;
         if (syncCob & 0x40000000) StartSyncProducer();
         else StartSyncConsumer();         
         StartTimeApplication();
         SetupEmcy();
      }
      initProcessOD();
      StartPDOs();
      if (startall) StartRemoteNode(0);
   }
   if (NmtState == stopped) {
      fprintf(stderr, "entering NMT stopped\n");
   }
   // interface to LocalNMTstateChange
   if (localNmtTask != NULL) {
      if (localNmtTask->running) {
         char reason[100];
         sprintf(reason, "realtime violation in task %20s", localNmtTask->name);
         logError(0xff, reason);
      } else {
         localNmtTask->statics[0] = NmtState;
         stdWakeup(localNmtTask);
      }
   }
   firstBoot = 0;
   // printf("NmtChngd finished\n");
}

static int startBus(int32_t BaudRate, int32_t id) {
   NmtId = id; NmtState = boot;
   init_lock();
   InitCan(BaudRate, CanError);
   initEvents();
   NmtChngd(unknown, resetAppl);
   return 1;
}

static int canRunning = 0;

int doCANopen(int32_t procNo, ComponentInst *inst, int32_t **PC, int32_t **SP, int32_t **FP, int32_t *excSP, int32_t **excHndlrPC, int32_t **excHndlrSP, int32_t **excHndlrFP) {
   if (procNo != 0 && !canRunning) return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0xfd, "CANopen: not properly started");
   switch (procNo) {
      case 0: {   // PROCEDURE startBus(INTEGER BaudRate, INTEGER NMTid, STRING DcfName);
         char DcfName[128];
         int32_t NMTid, BaudRate; 

         buildFileName(DcfName, projDir, (char *)Memory[**SP], "");  
         fprintf(stderr, "CANopen: reading DCF %s\n", DcfName);
         (*SP)++;
         NMTid = **SP; (*SP)++;
         BaudRate = **SP; (*SP)++;
         FILE *DCF = fopen(DcfName, "rb");
         if (DCF == NULL) {
            char reason[256];
            sprintf(reason, "CANopen: problem starting, no such device configuration, %s", DcfName);
            return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0xfd, reason);
         }
         readConcise(DCF);
         fclose(DCF);
         if (!canRunning && startBus(BaudRate, NMTid)) canRunning = 1;
         else return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0xfd, "CANopen: problem starting");
         break;
      }
      case 1: {   // PROCEDURE TPDOwrite(INTEGER TPODno);
         int32_t pdoNo;
         pdoNo = **SP; (*SP)++;
         WritePDO(pdoNo);
         break;
      }
      case 2: {   // PROCEDURE TIMEwrite();
         int32_t t = getCanTimeStamp();
         if ((timeCobid & 0xC0000000) != 0x40000000) return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0xfd, "CANopen: not configured as time producer");
         WriteTime(t % 86400000, t / 86400000);                  
         break;
      }
      case 3: {   // PROCEDURE EMCYwrite(INTEGER EMCYcode);
         uint32_t emcyCode;
         emcyCode = (uint32_t)(**SP); (*SP)++;
         ReportEmcy(emcyCode);
         break;
      }
      case 4: {   // PROCEDURE startRemoteNode(INTEGER NodeId);
         int32_t ni;
         ni = **SP; (*SP)++;
         if (!IsNmtMaster() && ni != NmtId && !(NMTStartup & 0x4)) return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0xfd, "CANopen: not NMT master");         
         StartRemoteNode(ni);
         break;
      }
      case 5: {   // PROCEDURE stopRemoteNode(INTEGER NodeId);
         int32_t ni;
         ni = **SP; (*SP)++;
         if (!IsNmtMaster() && ni != NmtId) return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0xfd, "CANopen: not NMT master");         
         StopRemoteNode(ni);
         break;
      }
      case 6: {   // PROCEDURE enterPreOperational(INTEGER NodeId);
         int32_t ni;
         ni = **SP; (*SP)++;
         if (!IsNmtMaster() && ni != NmtId) return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0xfd, "CANopen: not NMT master");         
         EnterPreOp(ni);
         break;
      }
      case 7: {   // PROCEDURE resetNode(INTEGER NodeId);
         int32_t ni;
         ni = **SP; (*SP)++;
         if (!IsNmtMaster() && ni != NmtId) return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0xfd, "CANopen: not NMT master");         
         ResetNode(ni);
         break;
      }
      case 8: {   // PROCEDURE resetCommunication(INTEGER NodeId);
         int32_t ni;
         ni = **SP; (*SP)++;
         if (!IsNmtMaster() && ni != NmtId) return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0xfd, "CANopen: not NMT master");         
         ResetCom(ni);
         break;
      }
      case 9: {   // PROCEDURE getNmtState(): INTEGER;
         (*SP--);
         **SP = NmtState;
         break;
      }
   }
   return 1;
}


