//   CanOpen, Technikum Joanneum GmbH
//   Peter Hintenaus
//   December 2000, 2009
//   Communication objects (PDO's, PDO mappings, SDOs, SYNC, TIME, EMCYs) according to DS301 V4.0
//
//   Change history:
//

#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <pthread.h>
#include <signal.h>
#include <limits.h>
#include "CanBasic.h"
#include "CanTimeStamp.h"
#include "Priorities.h"
#include "ComObj.h"
#include "nmt.h"
#include "ObjDict.h"
#include "Event.h"
#include "error.h"
#include "size.h"
#include "../vm/vm.h"

//-------------------------------- PDOs -----------------------------------------

TPdoType  TPDO[TPDONO];
RPdoType  RPDO[RPDONO];

static TPdoType *syncXmitPdos = NULL;

static void WritePdoCB(union sigval p) {
   WritePDO(p.sival_int);
}

// action to reenable a PDO after its inhibitTime has passed  
static void ReEnable(union sigval p) {
   TPdoType *pdo = &TPDO[p.sival_int];
   take_lock();
   if (pdo->state == xmitreq) {
      pdo->state = enabled;
      WritePDO(p.sival_int);
   } else {
      pdo->state = enabled;
      if (pdo->eventTimer > 0) reSchedule(&(pdo->eventTimerEvent), pdo->eventTimer * 10 - pdo->inhibitTime, 10000);
   }
   release_lock();
}

static void SyncRcvCallback(MobType *m);
static void SyncXmtCallback(MobType *m);
static void ScheduleSyncXmit(TPdoType *pdo, uint8_t period);

// callback for xmit PDOs.
static void TPDOCallback(MobType *m) {
   int pdono = m->usrData;
   TPdoType *pdo = &(TPDO[pdono]);
   if (pdo->inhibitTime != 0) {
      pdo->state = inhibited;
      reSchedule(&(pdo->inhibitEvent), pdo->inhibitTime, 10000);
   } else if (pdo->eventTimer > 0) reSchedule(&(pdo->eventTimerEvent), pdo->eventTimer, 1000);
   if (pdo->task != NULL) {
      if (pdo->task->running) {
         char reason[100];
         sprintf(reason, "realtime violation in task %20s", pdo->task->name);
         logError(0xff, reason);
      }
      pdo->task->statics[0] = getCanTimeStamp();
      stdWakeup(pdo->task);
   }
}

static void RPDOCallback(MobType *m) {
   int pdono = m->usrData;
   int32_t ts = getCanTimeStamp();
   RPdoType *pdo = &(RPDO[pdono]); 

   if (pdo->task == NULL) {
      if (!UnPack(m->data.c, m->datalen, &(RPdoMapping[pdono]), ts)) ReportEmcy(0x8210);
   } else {
      pthread_mutex_lock(&(pdo->task->mutex));
      if (pdo->cnt == pdo->siz) {
         char reason[100];
         pthread_mutex_unlock(&(pdo->task->mutex));
         sprintf(reason, "RPDO data loss in task %20s", pdo->task->name);
         logError(0xff, reason);
      } else {
         pdo->msg[pdo->in].data.i = m->data.i;
         pdo->msg[pdo->in].datalen = m->datalen;
         pdo->msg[pdo->in].ts = ts;
         (pdo->in)++;
         if (pdo->in == pdo->siz) pdo->in = 0;
         (pdo->cnt)++;
         if (pdo->cnt == 1) {
            pdo->task->running = 1;      
            pthread_cond_signal(&(pdo->task->cond));
         }
         pthread_mutex_unlock(&(pdo->task->mutex));
      }
   }
}

void StartPDOs(void) {
   int i, j, l;

   // setup TPDOs
   for (i = 0; i < TPDONO; i++) { 
      if (!(TPDO[i].cobid & 0x80000000)) {
         // PDO is valid
         for (l = 0, j = 0; j < TPdoMapping[i].no; j++) l += TPdoMapping[i].entry[j].length;
         TPdoMapping[i].bitsize = l;
         // printf("StartPDOs TPDO %d: cobId = 0x%x, mode = 0x%x\n", i, TPDO[i].cobid, TPDO[i].xmitTy);
         take_lock();
         if (TPDO[i].xmitTy < 241) TPDO[i].mob = InitMob(TPDO[i].cobid, sftxmit, i, SyncXmtCallback); 
         else TPDO[i].mob = InitMob(TPDO[i].cobid, sftxmit, i, TPDOCallback);
         if (TPDO[i].inhibitTime != 0) SetupEvent(&TPDO[i].inhibitEvent, ReEnable, i);
         if (TPDO[i].eventTimer > 0) SetupEvent(&TPDO[i].eventTimerEvent, WritePdoCB, i); 
         TPDO[i].state = enabled;
         TPDO[i].nxt = NULL;
         TPDO[i].delay = 0xFF;
         release_lock();
      }
   }
   for (i = 0; i < RPDONO; i++) {
      if (!(RPDO[i].cobid & 0x80000000)) {
         // PDO is valid
         for (l = 0, j = 0; j < RPdoMapping[i].no; j++) l += RPdoMapping[i].entry[j].length;
         RPdoMapping[i].bitsize = l;
         take_lock();
         if (RPDO[i].xmitTy < 241) RPDO[i].mob = InitMob(RPDO[i].cobid, sftrcv, i, SyncRcvCallback);
         else RPDO[i].mob = InitMob(RPDO[i].cobid, sftrcv, i, RPDOCallback);
         release_lock();
      }
   }
}

void StopPDOs(void) {
   int i;

   take_lock();
   syncXmitPdos = NULL;
   release_lock();

   // TPDOs
   for (i = 0; i < TPDONO; i++) { 
      if (!(TPDO[i].cobid & 0x80000000)) {
         // PDO is valid
         take_lock();
         DeleteMob(TPDO[i].cobid);
         if (TPDO[i].eventTimer > 0) cancel(&TPDO[i].eventTimerEvent);
         if (TPDO[i].inhibitTime != 0) cancel(&TPDO[i].inhibitEvent);
         release_lock();
      }
   }
   for (i = 0; i < RPDONO; i++) {
      if (!(RPDO[i].cobid & 0x80000000)) {
         // PDO is valid
         DeleteMob(RPDO[i].cobid);
      }
   }
}

static void DummyMobCallback(MobType  *foo) {}

static void DummyCallback(int foo) {}

void WritePDO(int pdono) {
   uint8_t datalen;
   char data[8];
   TPdoType *pdo = TPDO + pdono;

   // printf("WritePDO, no = %d, cobid = 0x%x, mode = 0x%x\n", pdono, pdo->cobid, pdo->xmitTy);
   if (NmtState != operational) return;
   if (pdo->xmitTy == 0){
      take_lock();
      if (pdo->nxt == NULL) ScheduleSyncXmit(pdo, 1);
      release_lock();
   } else if (pdo->xmitTy <= 253) {
      take_lock();       
      ScheduleSyncXmit(pdo, pdo->xmitTy);
      release_lock(); 
   } else {
      take_lock();
      if (pdo->state == enabled) {
         Pack(data, &datalen, TPdoMapping + pdono, getCanTimeStamp());
         SetData(pdo->mob, datalen, data);
         XmitMsg(pdo->mob);
      } else {
         TPDO[pdono].state = xmitreq;
      }
      release_lock();
   }
}

void ReadPDO(int RPDOno) {
   XmitRmtFrm(RPDO[RPDOno].mob);
}

// ------------------------- PDO Mappings -------------------------------------

PdoMapping  TPdoMapping[TPDONO];
PdoMapping  RPdoMapping[RPDONO];

void InitPdoMapping(PdoMapping  *m) {
   int i;
   m->no = 0;
   for (i = 0; i < 0x40; i++) m->entry[i].subindex = m->entry[i].index = 0;      // invalidate entries
}

static const uint8_t mask[] = {0x01, 0x03, 0x07, 0x0f, 0x1f, 0x3f, 0x7f};

static uint32_t min3(uint32_t i1, uint32_t i2, uint32_t i3) {
   if (i1 <= i2 && i1 <= i3) return i1;
   if (i2 <= i1 && i2 <= i3) return i2;
   return i3;
}

void Pack(char *data, uint8_t *datalen, PdoMapping  *m, int32_t TimeStamp) {
   unsigned bitpos = 0, bytepos = 0;
   int i;

   *data = 0; 
   READLOCK;
   for (i = 0; i < m->no; i++) {
      unsigned srcbitpos = 0;
      OdEntry *src;
      char *srcdat;
      uint8_t st;
      PdoMappingEntry e = m->entry[i];
      int dl, sgnxtnd;
      GetProcEntry(e.index, e.subindex, &st, &dl, &src, &sgnxtnd);
      srcdat = (char *)(&(src->Data));
      if (bitpos == 0 && srcbitpos == 0 && (e.length & 0x7) == 0) {
         // everything is byte aligned, copy
         char  *l = srcdat + (e.length >> 3);
         while (srcdat < l) {
            *data = *srcdat; 
            data++; srcdat++; bytepos++;
         }
         *data = 0;
      } else {
         unsigned nobits2cpy = e.length;
         char srcbyte = *srcdat;
         while (nobits2cpy > 0) {
            uint32_t nbits = min3(nobits2cpy, 8 - bitpos, 8 - srcbitpos);
            *data |= ((srcbyte >> srcbitpos) & mask[nbits - 1]) << bitpos;
            bitpos += nbits;
            if (bitpos == 8) {
               data++; bytepos++;
               if (bytepos < 8) *data = 0;
               bitpos = 0;
            }
            srcbitpos += nbits;
            if (srcbitpos == 8) {
               srcdat++;
               srcbyte = *srcdat;
               srcbitpos = 0;
            }
            nobits2cpy -= nbits;
         }         
      }
      src->TransferTime = TimeStamp;
   }
   UNLOCK;
   if (bitpos != 0) *datalen = bytepos + 1; else *datalen = bytepos;
}
   
int UnPack(char *data, uint8_t datalen, PdoMapping *m, int32_t TimeStamp) {
   unsigned srcbitpos = 0;
   int i;

   if ((m->bitsize + 7) / 8 != datalen) return 0;
   WRITELOCK;
   for (i = 0; i < m->no; i++) {
      unsigned bitpos = 0;
      OdEntry *ODE;
      uint8_t *dest;
      PdoMappingEntry e = m->entry[i];
      int dl, sgnxtnd; 
      uint8_t st;
      GetProcEntry(e.index, e.subindex, &st, &dl, &ODE, &sgnxtnd);
      dest = (char *)(&(ODE->Data));

      if (bitpos == 0 && srcbitpos == 0 && (e.length & 0x7) == 0) {
         // everything is byte aligned, copy
         uint8_t *l = dest + (e.length >> 3);
         while (dest < l) {
            *dest = *data; dest++; data++;
         }
      } else {
         unsigned nobits2cpy = e.length;
         while (nobits2cpy > 0) {
            unsigned nbits = min3(nobits2cpy, 8 - bitpos, 8 - srcbitpos);
            *dest &= ~(mask[nbits - 1] << bitpos);
            *dest |= ((((*data) >> srcbitpos) & mask[nbits - 1]) << bitpos);
            bitpos += nbits;
            if (bitpos == 8) {
               dest++;
               bitpos = 0;
            }
            srcbitpos += nbits;
            if (srcbitpos == 8) {
               data++;
               srcbitpos = 0;
            }
            nobits2cpy -= nbits;
         }         
      }
      if (sgnxtnd) ODE->Data = ((ODE->Data) << (32 - e.length)) >> (32 - e.length);
      ODE->TransferTime = TimeStamp;
   }
   UNLOCK;
   return 1;
}

// ---------------------------------- SDOs ----------------------------------------- 

SdoTy ClientSdo[CSDONO];
SdoTy ServerSdo[SSDONO];

static void AbortSdoXfer(SdoTy *sdo, ComModelTy who, int indicate, uint32_t reason) {
   char msg[8];
   MobIdTy mob = (who == server) ? sdo->S2Cmob : sdo->C2Smob; 
   msg[0] = 0x80;
   msg[1] = (char)(sdo->idx & 0xff);
   msg[2] = (char)(sdo->idx >> 8);
   msg[3] = sdo->subidx;
   msg[4] = (char)(reason & 0xff);
   msg[5] = (char)((reason >> 8) & 0xff);
   msg[6] = (char)((reason >> 16) & 0xff);
   msg[7] = (char)((reason >> 24) & 0xff);
   SetData(mob, 8, msg);
   sdo->state = soondone;
   XmitMsg(mob);
   sdo->result = reason;
   if (indicate) sdo->result = ServerSdoIndication(sdo, reason);
   else {
      pthread_mutex_lock(&sdo->mutex);
      sdo->state = done;
      pthread_cond_broadcast(&sdo->cond);
      pthread_mutex_unlock(&sdo->mutex);
   }   
}

static void ServerTimeout(union sigval s) {
   SdoTy *sdo = &(ServerSdo[s.sival_int]);
   take_lock();
   AbortSdoXfer(sdo, server, 1, SDOTIMEOUTERR);
   release_lock();
}

// these are mob callbacks, therefore they are locked...
static void Server2CCallback(MobType  *m) { // setup timeout
   // printf("SDO Server: Server -> Client sent\n");
   SdoTy *sdo = &(ServerSdo[m->usrData]);
   if (sdo->state == upld || sdo->state == dwnld) reSchedule(&(sdo->event), SDOTIMEOUT, 1000);   
}

static void C2ServerCallback(MobType  *mob) {
   SdoTy *sdo = &(ServerSdo[mob->usrData]);
   char resp[8] = {0, 0, 0, 0, 0, 0, 0, 0};
   uint32_t reason;
   int len, i;

   // printf("SDO Server: Client -> Server, cmd: %x\n", (int)(mob->data.c[0]));
   if (sdo->state != idle) cancel(&(sdo->event));
   if (mob->datalen != 8) {
      if (sdo->state == idle) {
         return;
      }
      AbortSdoXfer(sdo, server, 1, SDOGENERROR);
      return;
   }

   switch (mob->data.c[0] & 0xe0) {
   case 0x80:                                        // Abort Sdo xfer received
      if (sdo->state != idle) {
         reason = (uint32_t)(mob->data.i >> 32);   // little endian!
         sdo->state = idle;
         ServerSdoIndication(sdo, reason);         // Client has aborted, indicate to server
      }
      return;
   case 0x20:                                       // initiate download
      if (sdo->state != idle) {
         ServerSdoIndication(sdo, 0x08000000);      // the last SDO transfer has to be killed!!!
         sdo->state = idle;
      }
      sdo->idx = (uint16_t)mob->data.c[1] | ((uint16_t)mob->data.c[2] << 8);
      sdo->subidx = mob->data.c[3];
      switch(mob->data.c[0] & 0x3) {
      case 0x1:
         sdo->datalen = (uint32_t)(mob->data.i >> 32);   // little endian!
         if (mob->data.c[6] != '\0' || mob->data.c[7] != '\0') {
            AbortSdoXfer(sdo, server, 1, SDONOSTORELOC);
            return;
         }
         break;
      case 0x2:
         sdo->datalen = -1;
         break;
      case 0x3:
         sdo->datalen = 4 - ((mob->data.c[0] & 0x0c) >> 2);
         break;
      }
      if ((reason = CheckSdoDwnld(sdo)) != (uint32_t)0) {
         AbortSdoXfer(sdo, server, 1, reason);
         return;
      }
      if ((mob->data.c[0] & 0x3) == (char)0x3) { // expedited transfer
         for (i = 0; i < sdo->datalen; i++) sdo->data[i] = mob->data.c[4 + i];  // alignment unknown
         reason = ServerSdoIndication(sdo, 0);
         if (reason != 0x0) {
            AbortSdoXfer(sdo, server, 1, reason);
            return;
         }
      } else {
         sdo->state = dwnld;
         sdo->toggle = 0x0;
      }
      resp[0] = 0x60; resp[1] = (char)(sdo->idx & 0xff); resp[2] = (char)(sdo->idx >> 8);
      resp[3] = sdo->subidx;
      SetData(sdo->S2Cmob, 8, resp);
      XmitMsg(sdo->S2Cmob);
      break;
   case 0x40:               // initiate upload
       if (sdo->state != idle) {
         ServerSdoIndication(sdo, 0x08000000);   // kill last xfer
         sdo->state = idle;
      }
      sdo->idx = (uint16_t)mob->data.c[1] | ((uint16_t)mob->data.c[2] << 8);
      sdo->subidx = mob->data.c[3];
      if ((reason = CheckSdoUpld(sdo)) != 0) {
         AbortSdoXfer(sdo, server, 1, reason);
         return;
      }

      resp[1] = (char)(sdo->idx & 0xff); resp[2] = (char)(sdo->idx >> 8); resp[3] = sdo->subidx; 
      if (sdo->datalen <= 4) {   // expedited transfer
         resp[0] = 0x43 | ((4 - sdo->datalen) << 2);
         for (i = 0; i < sdo->datalen; i++) resp[4+i] = sdo->data[i];
      } else {
         resp[0] = 0x41;
         resp[4] = (char)(sdo->datalen & 0xff);  resp[5] = (char)(sdo->datalen >> 8);
         sdo->state = upld;
         sdo->toggle = 0;
      }
      SetData(sdo->S2Cmob, 8, resp);
      XmitMsg(sdo->S2Cmob);
      break;
   case 0x0:               // download message
      if (sdo->state != dwnld) {
         AbortSdoXfer(sdo, server, 1, SDOCOMMINVALID);
         return;
      }
      if ((mob->data.c[0] & 0x10) != sdo->toggle) {
         AbortSdoXfer(sdo, server, 1, SDONOTOGGLE);
         return;
      }
      len = 7 - ((mob->data.c[0] & 0x0e) >> 1);
      if (len > sdo->datalen) {
         AbortSdoXfer(sdo, server, 1, SDOLENERR);
         return;
      }
      if (mob->data.c[0] & 0x01) {
         if (len < sdo->datalen) { 
            AbortSdoXfer(sdo, server, 1, SDOLENERR);
            return;
         }
         len = sdo->datalen;
         sdo->state = idle;
      }
      for (i = 1; i <= len; i++) {
         *(sdo->data) = mob->data.c[i];  //alignment unknown
         sdo->data++;
      }
      sdo->datalen -= len;

      if (sdo->state == idle) {
         reason = ServerSdoIndication(sdo, 0);
         if (reason != 0x0) {
            AbortSdoXfer(sdo, server, 1, reason);
            return;
         }
      }
      resp[0] = 0x20 | sdo->toggle; 
      SetData(sdo->S2Cmob, 8, resp);
      XmitMsg(sdo->S2Cmob);
      sdo->toggle ^= 0x10;
      break;
   case 0x60:               // upload message
      if (sdo->state != upld) {
         AbortSdoXfer(sdo, server, 1, SDOCOMMINVALID);
         return;
      }
      if ((mob->data.c[0] & 0x10) != sdo->toggle) {
         AbortSdoXfer(sdo, server, 1, SDONOTOGGLE);
         return;
      }
      if (sdo->datalen <= 7) {
         len = sdo->datalen;
         resp[0] = sdo->toggle | (char)0x01 | (char)((7 - len) << 1);
         sdo->state = idle;
      } else {
         len = 7;
         resp[0] = sdo->toggle;
      }
      for (i = 1; i <= len; i++) {
         resp[i] = *(sdo->data); // alignment unknown
         sdo->data++;
      }
      sdo->datalen -= len;
      if (sdo->state == idle) {
         reason = ServerSdoIndication(sdo, 0);
         if (reason != 0) {
            AbortSdoXfer(sdo, server, 1, reason);
            return;
         }
      }
      SetData(sdo->S2Cmob, 8, resp);
      XmitMsg(sdo->S2Cmob);
      sdo->toggle ^= 0x10;
      break;
   default:
      AbortSdoXfer(sdo, server, 1, SDOCOMMINVALID);
   }
}

static void S2ClientCallback(MobType  *mob) { // handle response
   SdoTy *sdo = &(ClientSdo[mob->usrData]);
   char data[8] = {0, 0, 0, 0, 0, 0, 0, 0};
   int len, i;

   // printf("SDO Client: Server -> Client, cmd = %x\n", (int)(mob->data.c[0]));
   if (sdo->state != idle) cancel(&(sdo->event));

   if (mob->datalen != 8) {
      if (sdo->state == idle) return;
      AbortSdoXfer(sdo, server, 0, SDOGENERROR);
      return;
   }

   switch (mob->data.c[0] & 0xe0) {
   case 0x80:            //Abort SDO Xfer
      if (sdo->state != idle) {
         sdo->result = (uint32_t)(mob->data.i >> 32);   // little endian
         pthread_mutex_lock(&sdo->mutex);
         sdo->state = done;
         pthread_cond_broadcast(&sdo->cond);
         pthread_mutex_unlock(&sdo->mutex);
      }
      break;
   case 0x60:            //initiate download response
      if (sdo->state != dwnld) {
         AbortSdoXfer(sdo, client, 0, SDOCOMMINVALID);
         return;
      }
      if (((uint16_t)mob->data.c[1] | ((uint16_t)mob->data.c[2] << 8)) != sdo->idx || mob->data.c[3] != sdo->subidx) {
         AbortSdoXfer(sdo, client, 0, SDOPARAINCOMP);
         return;
      }
      if (sdo->datalen == 0) {   //expidited xfer
         sdo->state = soondone;
      } else {
         sdo->toggle = 0;
         if (sdo->datalen <= 7) {
            data[0] = 1 | ((7 - sdo->datalen) << 1);
            sdo->state = soondone;
            len = sdo->datalen;
         } else {
            data[0] = 0;
            len = 7;
         }
         for (i = 1; i <= len; i++) {
            data[i] = *(sdo->data);
            sdo->data++;
         }
         sdo->datalen -= len;
         SetData(sdo->C2Smob, 8, data);
         XmitMsg(sdo->C2Smob);
      }
      if (sdo->state == soondone) {
         sdo->result = 0;
         pthread_mutex_lock(&sdo->mutex);
         sdo->state = done;
         pthread_cond_broadcast(&sdo->cond);
         pthread_mutex_unlock(&sdo->mutex);
      }
      break;
   case 0x20:            //download SDO segment response
      if (sdo->state != dwnld) {
         AbortSdoXfer(sdo, client, 0, SDOCOMMINVALID);
         return;
      }
      if ((mob->data.c[0] & 0x10) != sdo->toggle) {
         AbortSdoXfer(sdo, client, 0, SDONOTOGGLE);
         return;
      }
      sdo->toggle ^= 0x10;
      if (sdo->datalen <= 7) {
         data[0] = 0x01 | sdo->toggle | ((7 - sdo->datalen) << 1);
         sdo->state = soondone;
         len = sdo->datalen;
      } else {
         data[0] = sdo->toggle;
         len = 7;
      }
      for (i = 1; i <= len; i++) {
         data[i] = *(sdo->data);
         sdo->data++;
      }
      sdo->datalen -= len;
      SetData(sdo->C2Smob, 8, data);
      XmitMsg(sdo->C2Smob);
      if (sdo->state == soondone) {
         sdo->result = 0;
         pthread_mutex_lock(&sdo->mutex);
         sdo->state = done;
         pthread_cond_broadcast(&sdo->cond);
         pthread_mutex_unlock(&sdo->mutex);
      }
      break;
   case 0x40:            //initiate upload response
      if (sdo->state != upld) {
         AbortSdoXfer(sdo, client, 0, SDOCOMMINVALID);
         return;
      }
      if (((uint16_t)mob->data.c[1] | ((uint16_t)mob->data.c[2] << 8)) != sdo->idx || mob->data.c[3] != sdo->subidx) {
         AbortSdoXfer(sdo, client, 0, SDOPARAINCOMP);
         return;
      }
      switch (mob->data.c[0] & 0x03) {
      case 0x1:
         len = (uint32_t)(mob->data.i >> 32);   // little endian
         if (mob->data.c[6] != '\0' || mob->data.c[7] != '\0' || len > sdo->datalen) {
            AbortSdoXfer(sdo, client, 0, SDONOSTORELOC);
            return;
         }
         break;
      case 0x02:
      case 0x03:      // expedited xfer
         if (mob->data.c[0] & 0x01) len = 4 - ((mob->data.c[0] & 0x0c) >> 2);
         else { 
            len = sdo->datalen;
            if (len > 4) len = 4;
         }
         if (len > sdo->datalen) AbortSdoXfer(sdo, client, 0, SDOLENHI);
         else {
            for (i = 0; i < len; i++) sdo->data[i] = mob->data.c[4 + i];
            sdo->translen = len;
            sdo->result = 0;
            pthread_mutex_lock(&sdo->mutex);
            sdo->state = done;
            pthread_cond_broadcast(&sdo->cond);
            pthread_mutex_unlock(&sdo->mutex);
         }
         return;
      }
      sdo->toggle = 0;
      data[0] = 0x60 | sdo->toggle;
      SetData(sdo->C2Smob, 8, data);
      XmitMsg(sdo->C2Smob);
      break;
   case 0x0:            //upload SDO segment response
      if (sdo->state != upld) {
         AbortSdoXfer(sdo, client, 0, SDOCOMMINVALID);
         return;
      }
      if ((mob->data.c[0] & 0x10) != sdo->toggle) {
         AbortSdoXfer(sdo, client, 0, SDONOTOGGLE);
         return;
      }
      len = 7 - ((mob->data.c[0] & 0x0e) >> 1);
      if (len > sdo->datalen) {
           AbortSdoXfer(sdo, server, 0, SDOLENERR);
         return;
      }
      if (mob->data.c[0] & 0x01) {
         sdo->state = soondone;
      }
      for (i = 1; i <= len; i++) {
         *(sdo->data) = mob->data.c[i];
         sdo->data++;
         sdo->translen++;
         sdo->datalen--;
      }

      if (sdo->state == soondone) {
         sdo->result = 0;
         pthread_mutex_lock(&sdo->mutex);
         sdo->state = done;
         pthread_cond_broadcast(&sdo->cond);
         pthread_mutex_unlock(&sdo->mutex);
      } else {
         sdo->toggle ^= 0x10;
         data[0] = 0x60 | sdo->toggle;
         SetData(sdo->C2Smob, 8, data);
         XmitMsg(sdo->C2Smob);
      }
      break;
   default:            //unknown 
      AbortSdoXfer(sdo, client, 0, SDOCOMMINVALID);
   }
}

static void ClientTimeout(union sigval s) {
   SdoTy *sdo = &(ClientSdo[s.sival_int]);
   take_lock();
   AbortSdoXfer(sdo, client, 0, SDOTIMEOUTERR);
   release_lock();
}

static void Client2SCallback(MobType  *m) { // setup timeout
   SdoTy *sdo = &(ClientSdo[m->usrData]);
   // printf("SDO Client, Client -> Server sent\n");
   if (sdo->state == upld || sdo->state == dwnld) reSchedule(&(sdo->event), SDOTIMEOUT, 1000);
}

void SetupServerSdo(int sdoId) {
   SdoTy *sdo = &(ServerSdo[sdoId]);

   // printf("Setting up server SDO %d, Client -> Server: %x, Server -> Client: %x\n", sdoId, sdo->Client2Server, sdo->Server2Client);
   SetupEvent(&(sdo->event),  ServerTimeout, sdoId);
   sdo->S2Cmob = InitMob(sdo->Server2Client, sftxmit, sdoId, Server2CCallback);
   sdo->C2Smob = InitMob(sdo->Client2Server, sftrcv, sdoId, C2ServerCallback);
   if (pthread_mutex_init(&sdo->mutex, NULL)) fprintf(stderr, "SetupServerSdo, pthread_mutex_init");
   if (pthread_cond_init(&sdo->cond, NULL)) fprintf(stderr, "SetupServerSdo, pthread_cond_init");
   sdo->state = idle;
}

void SetupClientSdo(int sdoId) {
   SdoTy *sdo = &(ClientSdo[sdoId]);

   SetupEvent(&(sdo->event), ClientTimeout, sdoId);
   sdo->S2Cmob = InitMob(sdo->Server2Client, sftrcv, sdoId, S2ClientCallback);
   sdo->C2Smob = InitMob(sdo->Client2Server, sftxmit, sdoId, Client2SCallback);
   if (pthread_mutex_init(&sdo->mutex, NULL)) fprintf(stderr, "SetupClientSdo, pthread_mutex_init");
   if (pthread_cond_init(&sdo->cond, NULL)) fprintf(stderr, "SetupClientSdo, pthread_cond_init");
   sdo->state = idle;
}

static void startSdoDwnLoad(SdoTy *sdo, char *data, uint16_t size, uint16_t idx, uint8_t subidx) {
   char msg[8] = {0, 0, 0, 0, 0, 0, 0, 0};
   int i;

   sdo->data = data;
   sdo->datalen = size;
   sdo->idx = idx;
   sdo->subidx = subidx;
   pthread_mutex_lock(&sdo->mutex);
   while (sdo->state != idle) pthread_cond_wait(&sdo->cond, &sdo->mutex);
   sdo->state = dwnld;
   pthread_mutex_unlock(&sdo->mutex);

   msg[1] = (char)(idx & 0xff);
   msg[2] = (char)(idx >> 8);
   msg[3] = subidx;
   if (size <= 4) {         // expedited xfer
      msg[0] = 0x23 | ((4 - size) << 2);
      for ( i = 0; i < size; i++) msg[4 + i] = data[i];
      sdo->datalen = 0;
   } else {
      msg[0] = 0x21;
      msg[4] = (char) (size & 0xff);
      msg[5] = (char) (size >> 8);
   }
   SetData(sdo->C2Smob, 8, msg);
   XmitMsg(sdo->C2Smob);
}

uint32_t SdoDwnLoad(SdoTy *sdo, char *data, uint16_t size, uint16_t idx, uint8_t subidx) {
   uint32_t r;

   startSdoDwnLoad(sdo, data, size, idx, subidx);

   pthread_mutex_lock(&sdo->mutex);
   while(sdo->state != done) pthread_cond_wait(&sdo->cond, &sdo->mutex);
   r = sdo->result;
   sdo->state = idle;
   pthread_cond_broadcast(&sdo->cond);
   pthread_mutex_unlock(&sdo->mutex);
   return r;
}

static void startSdoUpload(SdoTy *sdo, char *data, uint16_t size, uint16_t idx, uint8_t subidx) {
   char msg[8] = {0, 0, 0, 0, 0, 0, 0, 0};
   
   sdo->data = data;
   sdo->datalen = size;
   sdo->translen = 0;
   sdo->idx = idx;
   sdo->subidx = subidx;
   pthread_mutex_lock(&sdo->mutex);
   while (sdo->state != idle) pthread_cond_wait(&sdo->cond, &sdo->mutex);
   sdo->state = upld;
   pthread_mutex_unlock(&sdo->mutex);
   msg[0] = 0x40;
   msg[1] = (char)(idx & 0xff);
   msg[2] = (char)(idx >> 8);
   msg[3] = subidx;
   SetData(sdo->C2Smob, 8, msg);
   XmitMsg(sdo->C2Smob);
}

uint32_t SdoUpload(SdoTy *sdo, char *data, uint16_t size, uint16_t idx, uint8_t subidx) {
   uint32_t r;

   startSdoUpload(sdo, data, size, idx, subidx);

   pthread_mutex_lock(&sdo->mutex);
   while(sdo->state != done) pthread_cond_wait(&sdo->cond, &sdo->mutex);
   r = sdo->result;
   sdo->state = idle;
   pthread_cond_broadcast(&sdo->cond);
   pthread_mutex_unlock(&sdo->mutex);
   return r;
}

// ------------------------------------- SYNCs -------------------------------------

static MobIdTy syncMob;

static int32_t syncRcvPdos = 0, syncRcvCnt = 0;

static struct {
   union {
      uint64_t i;
      uint8_t c[8];
   } data;
   uint32_t datalen;
   int32_t ts;
   int pdono;
} syncRcvBuffer[2 * RPDONO];

EventTy SyncEvent;
EventTy SyncWindowEvent;
static int syncTicking = 0, syncWindowTicking = 0;

// callback for xmit PDOs.
static void SyncXmtCallback(MobType  *m) {
   int pdono = m->usrData;
   TPdoType *pdo = &(TPDO[pdono]);
   if (pdo->task != NULL) {
      if (pdo->task->running) {
         char reason[100];
         sprintf(reason, "realtime violation in task %20s", pdo->task->name);
         logError(0xff, reason);
      }
      pdo->task->statics[0] = getCanTimeStamp();
      stdWakeup(pdo->task);
   }
}

static void SyncRcvCallback(MobType  *m) {
   int pdono = m->usrData;
   RPdoType *pdo = &(RPDO[pdono]);
   int bi;

   if (pdo->syncIdx == -1 || pdo->syncIdx < syncRcvPdos || syncRcvPdos + syncRcvCnt - 1 < pdo->syncIdx) {
      bi = pdo->syncIdx = syncRcvPdos + syncRcvCnt;
      syncRcvCnt++;
   } else
      bi = pdo->syncIdx;
   if (bi >= 2 * RPDONO) bi -= 2 * RPDONO;

   // enter pdo into syncRcvBuffer
   syncRcvBuffer[bi].ts = getCanTimeStamp();
   syncRcvBuffer[bi].data.i = m->data.i;
   syncRcvBuffer[bi].datalen = m->datalen;
   syncRcvBuffer[bi].pdono = pdono;
}

static void ScheduleSyncXmit(TPdoType *pdo, uint8_t period) {
   if (pdo->delay != 0xFF) return;
   pdo->delay = period;
   if (syncXmitPdos == NULL) {
      pdo->nxt = NULL;
      syncXmitPdos = pdo;
   } else if (syncXmitPdos->delay > period) {
      syncXmitPdos->delay -= period;
      pdo->nxt = syncXmitPdos;
      syncXmitPdos = pdo;
   } else {
      TPdoType *prv = syncXmitPdos, *nxt;
      for (;;) {
         period -= prv->delay;
         if ((nxt = prv->nxt) == NULL) break;
         if (period < nxt->delay) break;
         prv = nxt;
      }
      pdo->delay = period; 
      pdo->nxt = nxt;
      if (nxt != NULL) nxt->delay -= period;
      prv->nxt = pdo;
   }
}

// mapped variables
CobIdTy syncCob; 
uint32_t comCycleTime;
uint32_t syncWindowLen;

// SyncWindowCallback:   Callback used to receive synch. RPDO's. 

vm *syncTask = NULL, *syncWindowTask = NULL;

static void SyncWindowCallback() {
   syncWindowTicking = 0;
   if (syncWindowTask != NULL) {
      if (syncWindowTask->running) {
         char reason[100];
         sprintf(reason, "realtime violation in task %20s", syncWindowTask->name);
         logError(0xff, reason);
      } else {
         syncWindowTask->statics[0] = getCanTimeStamp();
         stdWakeup(syncWindowTask);
      }
   }
}

static pthread_t Syncer;
static pthread_cond_t SyncCond;    // Condition we are waiting for
static pthread_mutex_t SyncMutex;  // Mutex for condition
static int syncOccured = 0;
static pthread_attr_t SyncAttrs;
static struct sched_param sp;   

// actual handling of synchronous data is delegated to a seperate thread. This is its code.
static void *doSyncer() {
   int32_t ts;

   for (;;) {
      int32_t syncRcvs, syncCnt, i;
      TPdoType *sXmit, *p;
      if (pthread_mutex_lock(&(SyncMutex))) {
         fprintf(stderr, "doSyncer, pthread_mutex_lock"); return;
      }
      if (syncOccured == -1) break;
      syncOccured = 0;
      while(!syncOccured) {
         if (pthread_cond_wait(&SyncCond, &SyncMutex)) {
            fprintf(stderr, "doSyncer, pthread_cond_wait");
            pthread_mutex_unlock(&(SyncMutex)); return;
         }
      }
      if (syncOccured == -1) break;
      pthread_mutex_unlock(&SyncMutex);
      // printf("in Syncer, started...\n");
      ts = getCanTimeStamp();
      // unpack SYNC data
      take_lock();
      syncRcvs = syncRcvPdos; syncCnt = syncRcvCnt;      
      syncRcvPdos += syncCnt;
      if (syncRcvPdos >= 2 * RPDONO) syncRcvPdos -= 2 * RPDONO;
      syncRcvCnt = 0;
      release_lock();
      // printf("in Syncer, syncRcvCnt = %d\n", syncCnt);
      if (syncTask != NULL && syncTask->running) {
         char reason[100];
         sprintf(reason, "realtime violation in task %20s", syncTask->name);
         logError(0xff, reason);
      }
      for (i = 0; i < syncCnt; i++) {
         int bi = syncRcvs + i;
         if (bi >= 2 * RPDONO) bi -= 2 * RPDONO;
         if (UnPack(syncRcvBuffer[bi].data.c, syncRcvBuffer[bi].datalen, RPdoMapping + syncRcvBuffer[bi].pdono, syncRcvBuffer[bi].ts)) {
            RPdoType *pdo = &(RPDO[syncRcvBuffer[bi].pdono]);
            take_lock();
            if (syncRcvs <= pdo->syncIdx && pdo->syncIdx <= syncRcvs + syncCnt - 1) pdo->syncIdx = -1;
            release_lock();
            if (pdo->task != NULL) {
               if (pdo->task->running) {
                  char reason[100];
                  sprintf(reason, "realtime violation in task %20s", pdo->task->name);
                  logError(0xff, reason);
               } else {
                  pdo->task->statics[0] = ts;
                  stdWakeup(pdo->task);
               }
            }
         } else ReportEmcy(0x8210);
      }
      // pack & send SYNC data
      // printf("in Syncer, starting to send data\n");
      take_lock();
      sXmit = syncXmitPdos;
      if (syncXmitPdos != NULL && syncXmitPdos->delay > 0) syncXmitPdos->delay--;
      p = NULL;
      while (syncXmitPdos != NULL && syncXmitPdos->delay == 0) {
         p = syncXmitPdos; syncXmitPdos = syncXmitPdos->nxt;
      }   
      release_lock();
      // printf("in Syncer, start syncXmitPdos = %x, end syncXmitPdos = %x\n", (int)sXmit, (int)p);
      if (p != NULL) p->nxt = NULL;
      while (sXmit != NULL) {
         uint8_t datalen;
         char data[8];
         TPdoType *nxt;
         Pack(data, &datalen, TPdoMapping + (sXmit - TPDO), getCanTimeStamp());
         SetData(sXmit->mob, datalen, data);
         XmitMsg(sXmit->mob);
         sXmit->delay = 0xFF;
         nxt = sXmit->nxt;
         if (sXmit->xmitTy != 0) {
            take_lock();
            ScheduleSyncXmit(sXmit, sXmit->xmitTy);
            release_lock();
         }
         sXmit = nxt;
      }
      if (syncTask != NULL) {
         syncTask->statics[0] = ts;
         stdWakeup(syncTask);
      }
   }
   pthread_mutex_unlock(&SyncMutex);
   if (pthread_mutex_destroy(&SyncMutex)) fprintf(stderr, "doSyncer, pthread_mutex_destroy");
   if (pthread_cond_destroy(&SyncCond)) fprintf(stderr, "doSyncer, pthread_cond_destroy");
   return NULL;
}

static void initSyncer() {
   int iret;

   sp.sched_priority = CANOPENSYNCP;
   iret = pthread_attr_init(&SyncAttrs);
   iret = pthread_attr_setschedpolicy(&SyncAttrs, SCHED_RR);   
   iret = pthread_attr_setschedparam(&SyncAttrs, &sp);
   iret = pthread_attr_setinheritsched(&SyncAttrs, PTHREAD_EXPLICIT_SCHED);
   iret = pthread_attr_setstacksize(&SyncAttrs, (0x20000 > PTHREAD_STACK_MIN) ? 0x20000 : PTHREAD_STACK_MIN);

   syncXmitPdos = NULL;  syncRcvCnt = syncRcvPdos = 0;
   syncOccured = 0;
   if (pthread_cond_init(&SyncCond, NULL)) {
      fprintf(stderr, "initSyncer, pthread_cond_init");
   }
   if (pthread_mutex_init(&SyncMutex, NULL)) {
      fprintf(stderr, "initSyncer, pthread_mutex_init");
   }
   if (pthread_create(&Syncer, &SyncAttrs, doSyncer, NULL)) {
      fprintf(stderr, "initSyncer, pthread_create");
   }
}

static void SyncCallback() {
   // printf("SyncCallback\n");
   pthread_mutex_lock(&SyncMutex);
   if (syncOccured) logError(0xff, "CANopen: internal error in SYNC");
   else { 
      syncOccured = 1;      
      pthread_cond_signal(&SyncCond);
   }
   pthread_mutex_unlock(&SyncMutex);

   if (syncWindowLen != 1) {
      syncWindowTicking = 1;
      reSchedule(&SyncWindowEvent, syncWindowLen, 1000000);
   }
   // printf("SyncCallback done\n");
}

static void XmitSync() {
   // locks already...
   XmitMsg(syncMob);
   SyncCallback();
}

static void SyncXmitCallback() {
}

void StartSyncProducer() {
   // printf("==> SyncProducer\n");
   initSyncer();
   syncMob = InitMob(syncCob, sftxmit, 0, SyncXmitCallback);
   SetupEvent(&SyncEvent, XmitSync, 0);
   SetupEvent(&SyncWindowEvent, SyncWindowCallback, 0);
   SetData(syncMob, 0, NULL);
   syncTicking = 1;
   schedulePeriodic(&SyncEvent, comCycleTime - 1, comCycleTime, 1000000);      
}

void StartSyncConsumer() {
   // printf("==> SyncConsumer\n");  
   initSyncer();
   SetupEvent(&SyncWindowEvent, SyncWindowCallback, 0);
   syncTicking = 0;
   syncMob = InitMob(syncCob, sftrcv, 0, SyncCallback);
}

void StopSync() {
   // printf("==> StopSync\n");
   take_lock();
   pthread_mutex_lock(&SyncMutex);
   syncOccured = -1;      
   pthread_cond_signal(&SyncCond);
   pthread_mutex_unlock(&SyncMutex);

   pthread_join(Syncer, NULL);

   if (syncTicking) cancel(&SyncEvent);
   if (syncWindowTicking) cancel(&SyncWindowEvent);
   syncTicking = 0; syncWindowTicking = 0;
   DeleteMob(syncCob);
   release_lock();
}

// ------------------------------------- TIME --------------------------------------

static MobIdTy timeMob;

CobIdTy timeCobid; 

void StartTimeProducer(void) {
   // printf("StartTimeProducer\n");
   timeMob = InitMob(timeCobid, sftxmit, 0, DummyMobCallback);
}

vm *timeRcvTask = NULL;

static void timeReceiver(MobType  *timeMob) {
   if (timeRcvTask != NULL) {
      if (timeRcvTask->running) {
         char reason[100];
         sprintf(reason, "realtime violation in task %20s", timeRcvTask->name);
         logError(0xff, reason);
      } else {
         timeRcvTask->statics[0] = getCanTimeStamp();
         timeRcvTask->statics[1] = (uint32_t)(timeMob->data.i);
         timeRcvTask->statics[2] = (uint32_t)(timeMob->data.i >> 32) & 0xffff;
         stdWakeup(timeRcvTask);
      }
   }      
}   

void StartTimeConsumer(void) {
   timeMob = InitMob(timeCobid, sftrcv, 0, timeReceiver);
}

void WriteTime(uint32_t ms, uint32_t days) {
   union {
      uint8_t c[6];
      uint32_t i[2];
   } d;

   d.i[0] = (ms & 0x0fffffff);
   d.i[1] =  days;
   SetData(timeMob, 6, d.c);
   XmitMsg(timeMob);
}

void StopTime() {
   DeleteMob(timeCobid);
}

// ------------------------------------- EMCY --------------------------------------

// mapped variables
CobIdTy emcyCob;
uint16_t emcyInhibit = 0;
uint8_t errorRegister;
uint32_t PreDefError[254];
uint8_t PreDefErrorNo; 

uint8_t emcyin, emcyout;
static EventTy emcyEvent;
static TStateTy emcyMsgState;
static MobIdTy emcyMob;

void ReEnableEmcy(union sigval foo) {
   take_lock();
   if (emcyMsgState == xmitreq) {
      unsigned errorCode = PreDefError[emcyout];
      uint8_t d[] = {0, 0, 0, 0, 0, 0, 0, 0};
      if (emcyout != emcyin) {   // still EMCYs to send? Else its an all clear.
         d[0] = errorCode & 0xff;
         d[1] = (errorCode >> 8) & 0xff;
         d[2] = errorRegister;
      }
      SetData(emcyMob, 8, d);
      XmitMsg(emcyMob);
      emcyMsgState = inhibited;
      if (emcyout != emcyin) {
         emcyout++;
         if (emcyout == 254) emcyout = 0;
      }
   } else emcyMsgState = enabled;
   release_lock();
}

void EmcySent(MobType  *foo) {
   if (emcyInhibit != 0) reSchedule(&emcyEvent, emcyInhibit, 10000);
   else emcyMsgState = enabled;
}

static int pndEmcys, iEmcys[8];

void SetupEmcy() {
   PreDefErrorNo = emcyin = emcyout = 0;
   errorRegister = 0;
   SetupEvent(&emcyEvent, ReEnableEmcy, 0);
   emcyMsgState = enabled;
   emcyMob = InitMob(emcyCob, sftxmit, 0, EmcySent);
   pndEmcys = 0;
   iEmcys[0] = iEmcys[1] = iEmcys[2] = iEmcys[3] = iEmcys[4] = iEmcys[5] = iEmcys[6] = iEmcys[7] = 0;
}

void StopEmcy() {
   emcyMsgState = inhibited;
   DeleteMob(emcyCob);
}

static const uint8_t ErrorBitpos[] = 
   {9, 0, 1, 2, 3, 7, 7, 7, 5, 5, 9, 9, 9, 9, 9, 5};



void ReportEmcy(unsigned errorCode) {
   uint8_t d[8] = {0, 0, 0, 0, 0, 0, 0, 0};
   unsigned erridx = errorCode >> 12;

   // printf("reporting Emergency errorCode = %x erridx = %x, emcyin = %d\n", errorCode, erridx, emcyin);
   take_lock();
   pndEmcys++;
   if (PreDefErrorNo < 254) PreDefErrorNo++;
   PreDefError[emcyin] = errorCode;
   emcyin++; 
   if (emcyin == 254) emcyin = 0;
   if (ErrorBitpos[erridx] != 9) {
      errorRegister |= (0x01 << (ErrorBitpos[erridx])); iEmcys[ErrorBitpos[erridx]]++;
   }
   if (emcyMob != NULL && emcyMsgState == enabled) {
      d[0] = errorCode & 0xff;
      d[1] = (errorCode >> 8) & 0xff;
      d[2] = errorRegister;
      // printf("ReportEmcy, setting data, emcyMob = %x\n", emcyMob);
      SetData(emcyMob, 8, d);
      XmitMsg(emcyMob);
      emcyMsgState = inhibited;
      emcyout++;
      if (emcyout == 254) emcyout = 0;
   } else emcyMsgState = xmitreq;
   release_lock();
}

void ClearEmcy(unsigned errorCode) {
   uint8_t d[8] = {0, 0, 0, 0, 0, 0, 0, 0};
   unsigned erridx = errorCode >> 12;

   // printf("cleanring Emergency %x\n", errorCode);
   take_lock();
   if (ErrorBitpos[erridx] != 9) {
      if (iEmcys[ErrorBitpos[erridx]] != 0) iEmcys[ErrorBitpos[erridx]]--;
      if (iEmcys[ErrorBitpos[erridx]] == 0) errorRegister &= (0xfe << (ErrorBitpos[erridx]));
   }
   if (pndEmcys != 0) pndEmcys--;   
   if (pndEmcys == 0) {
      SetData(emcyMob, 8, d);
      if (emcyMsgState == enabled) {
         XmitMsg(emcyMob);
         emcyMsgState = inhibited;
      } else emcyMsgState = xmitreq;
   }
   release_lock();
}

consumedEmcyTy consumedEmcy[127];

void emcyCallback(MobType  *mob) {
   consumedEmcyTy *emcy = &(consumedEmcy[mob->usrData]);
   unsigned errorCode;

   if (mob->datalen != 8) return;

   errorCode = mob->data.c[0] | (((unsigned)mob->data.c[1]) << 8);

   if (emcy->task != NULL) {
      if (emcy->task->running) {
         char reason[100];
         sprintf(reason, "realtime violation in task %20s", timeRcvTask->name);
         logError(0xff, reason);
      } else {
         emcy->task->statics[0] = errorCode;
         emcy->task->statics[1] = mob->data.c[2];
         emcy->task->statics[2] = mob->data.c[3];
         emcy->task->statics[3] = mob->data.c[4];
         emcy->task->statics[4] = mob->data.c[5];
         emcy->task->statics[5] = mob->data.c[6];
         emcy->task->statics[6] = mob->data.c[7];
         stdWakeup(emcy->task);
      }
   }
}

void ConsumeEmcy(int16_t emcyid) {
   consumedEmcyTy *emcy = &(consumedEmcy[emcyid]);

   if (emcy->emcyCob & 0x80000000) return;
   emcy->mob = InitMob(emcy->emcyCob, sftrcv, emcyid, emcyCallback);
}

void initComObjEvents(void) {
   int i;

   for (i = 0; i < TPDONO; i++) {
      TPDO[i].inhibitEvent.inited = 0;
      TPDO[i].eventTimerEvent.inited = 0;
   }
   for (i = 0; i < CSDONO; i++) {
      ClientSdo[i].event.inited = 0;
   }
   for (i = 0; i < SSDONO; i++) {
      ServerSdo[i].event.inited = 0;
   }   
   SyncEvent.inited = 0;
   SyncWindowEvent.inited = 0;
   emcyEvent.inited = 0;
}


