#ifndef __comobj__
#define __comobj__

//   CanOpen
//   Peter Hintenaus 2000 - 2009, Gernot Kvas 2008
//   December 2000
//   Definition of communication objects according to DS301 V4.0
//
//   Change history:
//

#include <stdint.h>
#include "CanBasic.h"
#include "Event.h"
#include "../vm/vm.h"

typedef enum {
   client, server,
   producer, consumer,
   master, slave,
   nomodel
} ComModelTy;

void initComObjEvents(void);

// --------------------------------- PDOs -------------------------------------------

typedef enum {
   enabled, inhibited, xmitreq
} TStateTy;

typedef struct TPdoStruct {
   // mapped to object dicitonary entries
   CobIdTy cobid;
   uint16_t inhibitTime;
   uint16_t eventTimer;
   uint8_t xmitTy;
   // not mapped
   uint8_t delay;         // for queueing for sync xmit
   struct TPdoStruct *nxt;
   TStateTy state;      // to handle ihibittime
   EventTy inhibitEvent;
   EventTy eventTimerEvent;
   MobIdTy mob;
   vm *task;            // for event delivery
} TPdoType;

extern TPdoType TPDO[];

typedef struct {
   unsigned char datalen;
   union {
      uint64_t i;
      unsigned char c[8];
   } data;
   int32_t ts;
} RPDOBuffer;

typedef struct {
   // mapped to object dicitonary entries
   CobIdTy cobid;
   uint8_t xmitTy;
   // not mapped
   MobIdTy mob;
   vm *task;            // for event delivery
   int32_t in, out, cnt, siz;
   RPDOBuffer *msg;
   int syncIdx;
} RPdoType;

extern RPdoType RPDO[];

// to be called just before device goes operational. StartPDOs does not send any PDOs.
void StartPDOs(); 

void StopPDOs();

void WritePDO(int TPDOno);      // the data parameter in DS301 V4.1 does not make sense, so it's gone
void ReadPDO(int RPDOno);

// ------------------------------ PDO Mappings ------------------------------------

typedef   struct {
   // mapped
   uint8_t length, /* in bits */ subindex;
   uint16_t index;
} PdoMappingEntry;

typedef struct {
   uint8_t no;      // mapped to subindex 0
   uint8_t bitsize;
   PdoMappingEntry entry[0x40];
} PdoMapping;

extern PdoMapping TPdoMapping[];
extern PdoMapping RPdoMapping[];

void InitPdoMapping(PdoMapping *);

void Pack(char *data, uint8_t *datalen, PdoMapping *, int32_t);
int UnPack(char *data, uint8_t datalen, PdoMapping *, int32_t);

// ------------------------------------- SDOs --------------------------------------

#define SDOTIMEOUT 1000

typedef enum {idle, soondone, done, upld, dwnld} SdoStateTy;

typedef struct SdoStruct {
   // mapped entries
   CobIdTy Client2Server, Server2Client;
   // not mapped
   MobIdTy S2Cmob, C2Smob;
   uint16_t idx;
   uint8_t subidx;
   uint8_t toggle;
   char *data;
   uint16_t datalen, translen;
   pthread_mutex_t mutex;
   pthread_cond_t cond;
   uint32_t result;
   SdoStateTy state;
   EventTy event;
   // mapped
   uint8_t nodeId;
} SdoTy;

extern SdoTy ClientSdo[];
extern SdoTy ServerSdo[];

void SetupServerSdo(int sdoId);

void SetupClientSdo(int sdoId);

uint32_t SdoDwnLoad(SdoTy *sdo, char *data, uint16_t size, uint16_t idx, uint8_t subidx);

uint32_t SdoUpload(SdoTy *sdo, char *data, uint16_t size, uint16_t idx, uint8_t subidx);

// ------------------------------------- SYNCs -------------------------------------

extern CobIdTy syncCob;

extern vm *syncTask, *syncWindowTask;

void StartSyncProducer();

void StopSync();

void StartSyncConsumer();

// ------------------------------------- TIME --------------------------------------

extern CobIdTy timeCobid;

extern vm *timeRcvTask;

void StartTimeProducer(void);

void StartTimeConsumer(void);

void WriteTime(uint32_t ms, uint32_t days);

void StopTime();

// ------------------------------------- EMCY --------------------------------------

extern uint32_t PreDefError[254];
extern uint8_t PreDefErrorNo; 

extern uint8_t emcyin, emcyout;

void SetupEmcy();

void StopEmcy();

void ReportEmcy(unsigned errorCode);

void ClearEmcy(unsigned errorCode);

typedef struct {
   // mapped
   CobIdTy emcyCob;
   // not mapped
   MobIdTy mob;
   vm *task;
} consumedEmcyTy;

extern consumedEmcyTy consumedEmcy[];

void ConsumeEmcy(int16_t emcyid);

#endif
