#include <stdint.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/types.h>
#include <linux/spi/spidev.h>
#include <pthread.h>
#include <sched.h>

#include "mcp2515.h"
#include "Priorities.h"

#define MAX_LENGTH 32
// #define DEBUG

static void pabort(const char *s) {
   perror(s);
   abort();
}

/* TX/RX Buffers for SPI transfers */
static uint8_t mcp_buf[MAX_LENGTH];

static const char *spi_dev_ = "/dev/spi0";
static const char *uio_dev_ = "/dev/uio0";
static uint8_t mode;
static uint8_t bits = 8;
static unsigned long speed = 8333333;
static uint32_t delay;
static int spi_fd;
static int uio_fd;

mcp2515_chip *hw = NULL;

static int mcp_transfer(int fd, int length) {
   int ret = 0;

   struct spi_ioc_transfer tr = {
      .tx_buf = (unsigned long)mcp_buf,
      .rx_buf = (unsigned long)mcp_buf,
      .len = length,
      .delay_usecs = delay,
      .speed_hz = speed,
      .bits_per_word = bits,
   };
#ifdef DEBUG
   printf("Out: ");
   for (ret = 0; ret < length; ret++) {
      printf("%.2X ", mcp_buf[ret]);
   }
   puts("");
#endif
   ret = ioctl(fd, SPI_IOC_MESSAGE(1), &tr);

   if(ret < 0)
      pabort("transfer failed");
#ifdef DEBUG
   printf("In : ");
   for (ret = 0; ret < length; ret++) {
      printf("%.2X ", mcp_buf[ret]);
   }
   puts("");
#endif
}

int mcp_init(const char* spi_dev, const char* uio_dev) {
   int ret = 0;

   if(hw != NULL)
      pabort("mcp2515_chip already malloced");

   hw = malloc(sizeof(mcp2515_chip));

   if(hw == NULL)
      pabort("can't malloc mcp2515_chip");

   hw->spi_fd = open(spi_dev, O_RDWR);
   if (hw->spi_fd < 0)
      pabort("can't open spi device");

   ret = ioctl(hw->spi_fd, SPI_IOC_WR_MODE, &mode);
   if (ret == -1)
      pabort("can't set spi mode");
   ret = ioctl(hw->spi_fd, SPI_IOC_WR_BITS_PER_WORD, &bits);
   if (ret == -1)
      pabort("can't set bits per word");
   ret = ioctl(hw->spi_fd, SPI_IOC_WR_MAX_SPEED_HZ, &speed);   
   if (ret == -1)
      pabort("can't set max speed hz");

   hw->uio_fd = open(uio_dev, O_RDWR);
   if (hw->uio_fd < 0)
      pabort("can't open uio device");

   return ret;
}

void mcp_setbaud(uint32_t baud) {
   uint8_t my_buf[1];

   // Default setting for Fosc = 20 MHz
   // Tosc = 50 ns
   // Baudrate = 125 kHz
   // BRP = 500 ns = 2 * (0x4 + 1) * Tosc
   // Tq = 16 * BRP
   // SJW = 0x1 * TQ
   // PRSEG = (0x1 + 1) * TQ 
   // PS1 = 7 * TQ
   // PS2 = 6 * TQ
   // Baudrate = 1000 kHz
   // BRP = 100 ns = 2 * (0 + 1) * Tosc
   // Tq = 10 * BRP
   // SJW = 1 * TQ
   // PRSEG = (1 + 1) * TQ
   // PS1 = (3 + 1) * TQ
   // PS2 = (2 + 1) * TQ
   if (baud == 1000) {
      my_buf[0] = 0x82;
      mcp_write(MCP_CNF3, my_buf, 1);
      // mcp_read(MCP_CNF3, my_buf, 1);

      my_buf[0] = 0x99;
      mcp_write(MCP_CNF2, my_buf, 1);
      // mcp_read(MCP_CNF2, my_buf, 1);

      my_buf[0] = 0x00;
      mcp_write(MCP_CNF1, my_buf, 1);
      // mcp_read(MCP_CNF1, my_buf, 1);
   } else {
      my_buf[0] = 0x85;
      mcp_write(MCP_CNF3, my_buf, 1);
      // mcp_read(MCP_CNF3, my_buf, 1);

      my_buf[0] = 0xB1;
      mcp_write(MCP_CNF2, my_buf, 1);
      // mcp_read(MCP_CNF2, my_buf, 1);

      my_buf[0] = 0x04;
      mcp_write(MCP_CNF1, my_buf, 1);
      // mcp_read(MCP_CNF1, my_buf, 1);
   }
}

int mcp_start(void* (*isr)(void*)) {
   uint8_t my_buf[2];   
   int iret;
   struct sched_param sp;   

   sp.sched_priority = CANOPENISRP;
   iret = pthread_attr_init(&hw->isr_attr);
   iret = pthread_attr_setschedpolicy(&hw->isr_attr, SCHED_FIFO);   
   iret = pthread_attr_setschedparam(&hw->isr_attr, &sp);
   iret = pthread_attr_setinheritsched(&hw->isr_attr, PTHREAD_EXPLICIT_SCHED);

   iret = pthread_create(&hw->isr_thread, &hw->isr_attr, isr, NULL);

   my_buf[0] = 0x3F;
   mcp_write(MCP_CANINTE, my_buf, 1);
   //mcp_read(MCP_CANINTE, my_buf, 1);

   my_buf[0] = 0x00;
   mcp_write(MCP_CANCTRL, my_buf, 1);
   //mcp_read(MCP_CANCTRL, my_buf, 1);

   my_buf[0] = 0x00;
   mcp_write(MCP_CANSTAT, my_buf, 1);
   //mcp_read(MCP_CANSTAT, my_buf, 1);

   my_buf[0] = 0x64;
   mcp_write(MCP_RXBXCTRL(0), my_buf, 1);
   my_buf[0] = 0x60;
   mcp_write(MCP_RXBXCTRL(1), my_buf, 1);
#if 0
   my_buf[0] = 0xAA;
   my_buf[1] = 0xA0;
   mcp_write(MCP_TXBXSIDH(0), my_buf, 2);
   //mcp_read(MCP_TXBXSIDH(0), my_buf, 2);

   my_buf[0] = 0x02;
   mcp_write(MCP_TXBXDLC(0), my_buf, 1);
   //mcp_read(MCP_TXBXDLC(0), my_buf, 1);

   my_buf[0] = 0x55;
   my_buf[1] = 0xAA;
   mcp_write(MCP_TXBXD0(0), my_buf, 2);
   //mcp_read(MCP_TXBXD0(0), my_buf, 2);

   my_buf[0] = 0x64;
   mcp_write(MCP_RXBXCTRL(0), my_buf, 1);
   //mcp_read(MCP_RXBXCTRL(0), my_buf, 1);

   mcp_rts(0);
#endif
   //mcp_read(MCP_RXBXD0(0), my_buf, 8);

   return iret;
}



/* Length is length for data only */
void mcp_write(uint8_t reg, uint8_t* data, uint8_t length) {
   mcp_buf[0] = MCP_WRITE;
   mcp_buf[1] = reg;
   memcpy(mcp_buf + 2, data, length);
   mcp_transfer(hw->spi_fd, length + 2);
}

void mcp_load_tx(uint8_t buf, uint8_t *msg) {
   int l = (msg[4] & 0xF) + 6;
   mcp_buf[0] = MCP_LOAD_TX(buf);
   memcpy(mcp_buf+1, msg, 13);
   mcp_transfer(hw->spi_fd, l);
}

/* Length is length for data only */
void mcp_read(uint8_t reg, uint8_t* data, uint8_t length) {
   int tot_length = length + 2;
   // Clear Buffer - not really necessary
   memset(mcp_buf + 2, 0xFF, length);
   mcp_buf[0] = MCP_READ;
   mcp_buf[1] = reg;   
   // Transmit
   mcp_transfer(hw->spi_fd, tot_length);
   // Result is stored on mcp_buf
   memcpy(data, mcp_buf + 2, length);
}

uint8_t mcp_read_stat(void) {
   mcp_buf[0] = MCP_READ_STAT;
   mcp_buf[1] = 0xFF;
   mcp_transfer(hw->spi_fd, 2);
   return mcp_buf[1];
}

uint8_t mcb_read_rxbctrl(uint8_t buf_nr) {
   mcp_buf[0] = MCP_RXBXCTRL(buf_nr);
   mcp_buf[1] = 0xFF;
   mcp_transfer(hw->spi_fd, 2);
   return mcp_buf[1];
}

void mcp_read_rx(uint8_t buf_nr, uint8_t* data) {
   memset(mcp_buf + 1, 0xFF, 13);
   mcp_buf[0] = MCP_READ_RX(buf_nr);
   mcp_transfer(hw->spi_fd, 14);
   memcpy(data, mcp_buf + 1, 13);
}  

void mcp_bitmod(uint8_t reg, uint8_t msk, uint8_t data) {
   mcp_buf[0] = MCP_BITMOD;
   mcp_buf[1] = reg;
   mcp_buf[2] = msk;
   mcp_buf[3] = data;
   mcp_transfer(hw->spi_fd, 4);
}

uint32_t mcp_intstat() {
   uint8_t tmp;
   mcp_read(MCP_CANINTF, &tmp, 1);
   return tmp;
}

void mcp_reset() {
   mcp_buf[0] = 0xC0;
   mcp_transfer(hw->spi_fd, 1);
}

void mcp_rts(uint8_t buf_nr) {
   mcp_buf[0] = MCP_RTS(buf_nr);
   mcp_transfer(hw->spi_fd, 1);
}

void mcp_close() {
   int ret;

   pthread_join(hw->isr_thread, NULL);

   // More error handling here
   ret = close(hw->spi_fd);
   ret = close(hw->uio_fd);

   free(hw);
}

