/*****************************************************************************
 * Radar.c
 *****************************************************************************/

#include <21363.h>
#include <sru.h>
#include <math.h>
#include <signal.h>
#include <sysreg.h>
#include <filter.h>
#include <window.h>
#include <stdlib.h>
#include "can.h"

#define SAMPLERATE 48000
#define SCANSPERSEC 10
#define SAMPLEOFFSET 300
#define FFTLENGTH 4096

#define ADC_GAIN00 0x0000
#define ADC_GAIN03 0x0009
#define ADC_GAIN06 0x0012
#define ADC_GAIN09 0x001B
#define ADC_GAIN12 0x0024


#define ADC_GAIN ADC_GAIN00

segment ("seg_dacdata") static int ramp[2 * SAMPLERATE / SCANSPERSEC + 1];
segment ("seg_adcdata") static int din[2 * SAMPLERATE / SCANSPERSEC];

typedef struct {
   unsigned CPSP;
   unsigned ICSP;
   int IMSP;
   unsigned IISP;
} SPortTCB;

#define EEPROMSPICS SPIFLG0
#define CANSPICS SPIFLG3

void sendDAC(int dat) {
   int dummy;

   *pSPICTL = TIMOD1 | WL16 | MSBF | SPIMS | SPIEN;
   sysreg_bit_clr(sysreg_FLAGS, FLG1);
   *pTXSPI = dat;
   while ((*pSPISTAT & RXS) == 0);
   sysreg_bit_set(sysreg_FLAGS, FLG1);
   dummy = *pRXSPI;
}

void sendADC(int dat) {
   int dummy;

   *pSPICTL = TIMOD1 | WL16 | MSBF | SPIMS | SPIEN;
   sysreg_bit_clr(sysreg_FLAGS, FLG2);
   *pTXSPI = dat;
   while ((*pSPISTAT & RXS) == 0);
   sysreg_bit_set(sysreg_FLAGS, FLG2);
   dummy = *pRXSPI;
}

void initCoreClk(void) {
   int i;
   *pPMCTL |= PLLBP;
   *pPMCTL &= ~0x3F;
   *pPMCTL |= 12;
   for (i = 0; i < 4096; i++);
   *pPMCTL &= ~PLLBP;
}    
 
static void handleADC(int);

void initConverters(SPortTCB txTCBs[2], SPortTCB rxTCBs[2]) {
   int i;
   //disable PCG erarly to satisfy delay requirement...
   *pPCG_CTLA_0 &= 0x3FFFFFFF;
  	sysreg_bit_set(sysreg_FLAGS, FLG0O | FLG1O | FLG2O | FLG3O);
  	sysreg_bit_set(sysreg_FLAGS, FLG0 | FLG1 | FLG2 | FLG3);
 
   // DAC routing PCG_A is used for FS & CLK
   SRU(PCG_FSA_O, DAI_PB18_I);
   SRU(PCG_FSA_O, SPORT0_FS_I);
   SRU(PCG_CLKA_O, DAI_PB19_I);
   SRU(PCG_CLKA_O, SPORT0_CLK_I);
   SRU(SPORT0_DA_O, DAI_PB20_I);
   SRU(HIGH, PBEN18_I);
   SRU(HIGH, PBEN19_I);
   SRU(HIGH, PBEN20_I);
   // PCG_A inputs
   SRU(LOW, PCG_SYNC_CLKA_I);
   SRU(LOW, PCG_EXTA_I);
   SRU(LOW, MISCA2_I);
   SRU(LOW, MISCA3_I);
   // ADC routing
   SRU(LOW, PBEN17_I);
   SRU(LOW, PBEN16_I);
   SRU(LOW, PBEN15_I);
   SRU(DAI_PB17_O, SPORT1_DA_I);
   SRU(DAI_PB16_O, SPORT1_CLK_I);
   SRU(DAI_PB15_O, SPORT1_FS_I);
   SRU(LOW, PBEN14_I);   
   
   ramp[0] = 0;
   for (i = 0; i < SAMPLERATE / SCANSPERSEC; i++) {
      ramp[2 * i + 1] = -i * 447392;
      ramp[2 * i + 2] = 0;
   }
   *pSPIBAUD = 0x10;
   *pSPIFLG = 0;
   *pSPICTL = TIMOD1 | WL16 | MSBF | SPIMS | SPIEN;
   
   *pCPSP0A = 0;
   txTCBs[0].IISP = (unsigned)(ramp);
   txTCBs[0].IMSP = 1;
   txTCBs[0].ICSP = 2 * SAMPLERATE / SCANSPERSEC;
   txTCBs[0].CPSP = ((unsigned)&txTCBs[1].IISP) - 0x80000;
   txTCBs[1].IISP = (unsigned)(ramp + 2 * SAMPLERATE / SCANSPERSEC);
   txTCBs[1].IMSP = -1;
   txTCBs[1].ICSP = 2 * SAMPLERATE / SCANSPERSEC;
   txTCBs[1].CPSP = ((unsigned)&txTCBs[0].IISP) - 0x80000;
         
   *pSPCTL0 = SPEN_A | DTYPE1 | SLEN32 | FSR | SDEN_A | SCHEN_A | SPTRAN;    
   *pCPSP0A = ((unsigned)&txTCBs[0].IISP) - 0x80000;
   
   *pPCG_CTLA_0 = 256;
   *pPCG_CTLA_1 = 8;
   *pPCG_PW = 8;
   *pPCG_SYNC = 0;
   *pPCG_CTLA_0 = (1 << 31) | (1 << 30) | 256;

   sendDAC(0xFFFC);
   sendDAC(0xFFFE);
   
   // setup ADC
   sendADC(0x0080 | ADC_GAIN);
   sendADC(0x1040);
   sendADC(0x2070);   

   interrupt(SIG_P3, handleADC);

   *pCPSP1A = 0;
   rxTCBs[0].IISP = (unsigned)(din);
   rxTCBs[0].IMSP = 1;
   rxTCBs[0].ICSP = 2 * SAMPLERATE / SCANSPERSEC - 2 * SAMPLEOFFSET;
   rxTCBs[0].CPSP = (((unsigned)&rxTCBs[1].IISP) - 0x80000) | 0x80000;
   rxTCBs[1].IISP = (unsigned)(din + 2 * SAMPLERATE / SCANSPERSEC - 2 * SAMPLEOFFSET);
   rxTCBs[1].IMSP = 1;
   rxTCBs[1].ICSP = 2 * SAMPLEOFFSET;
   rxTCBs[1].CPSP = ((unsigned)&rxTCBs[0].IISP) - 0x80000;

   *pSPCTL1 = SPEN_A | DTYPE1 | SLEN32 | SDEN_A | SCHEN_A;    
   *pCPSP1A = ((unsigned)&rxTCBs[0].IISP) - 0x80000;
      
   // release ADC from powerdown
   sendADC(ADC_GAIN);
}

static volatile int state;

#define STATE_READY 0
#define STATE_PROCESS 1

#pragma align 4096
float dm I[FFTLENGTH];
#pragma align 4096
float pm Q[FFTLENGTH];

float dm TmpR[FFTLENGTH];
float pm TmpI[FFTLENGTH];

float dm twid_r[FFTLENGTH/2];
float pm twid_i[FFTLENGTH/2];

volatile int forwardScan;

float dm window[FFTLENGTH];

#define CMD_COBID 0x89

#define SIGMASK  0x1
#define SPECMASK 0x2

volatile int ComMsk = 0x2;
volatile int SpecLen = 200;
volatile int DwnSmpl = 4;

void handleCan(unsigned cobId, unsigned dlc, unsigned *data) {
   if (cobId == CMD_COBID && dlc == 8) {
      ComMsk = data[0] & 0xFF;
      SpecLen = (data[0] & 0xFFFF) >> 8;
      DwnSmpl = data[0] >> 24;
   }
}

void main() {
   SPortTCB tx[2], rx[2];
   int j, gain = 0;
//   initCoreClk();
   state = STATE_READY;

   twidfftf(twid_r,twid_i,FFTLENGTH);
   gen_hanning(window, 1, FFTLENGTH);
   initConverters(tx, rx);
   initCan(handleCan);
	sysreg_bit_set(sysreg_MODE1, IRPTEN);
   for (;;) {
      while (state == STATE_READY) /* idle() */;
      if ((ComMsk & SIGMASK) != 0) sendSignal(I, Q, FFTLENGTH, DwnSmpl);
            
      cfftf(I, Q, TmpR, TmpI, twid_r, twid_i, FFTLENGTH);
      
      for (j = 0; j < FFTLENGTH; j++) {
         float i = I[j], q = Q[j];
         I[j] = sqrtf(i * i + q * q);
         Q[j] = atan2f(q, i);
      }
      if ((ComMsk & SPECMASK) != 0) sendSpectrum(I, Q, FFTLENGTH, forwardScan, SpecLen);       
      state = STATE_READY;
   }
}

volatile int skipped = 0;

static void handleADC(int vector) {
   static int forward = 0;
   int i;
   int *pI = din + SAMPLEOFFSET * 2;
   int *pQ = din + SAMPLEOFFSET * 2 + 1;
   
   forward = ! forward;
   if (state != STATE_READY) {
      skipped++;
      return;
   }
   for (i = 0; i < FFTLENGTH; i++) {
      float w = window[i];
      I[i] = ((*pI) >> 8) * w;
      Q[i] = ((*pQ) >> 8) * w;
      pI += 2; pQ += 2;
   }
   forwardScan = forward;
   state = STATE_PROCESS;
}
