#include <21363.h>
#include <sysreg.h>
#include <sru.h>
#include "can.h"
#include "mcp2515.h"

#define SPEC_A_HDR_COBID  0x90
#define SPEC_A_DATA_COBID 0x91
#define SPEC_P_HDR_COBID  0x92
#define SPEC_P_DATA_COBID 0x93
#define SIG_I_HDR_COBID   0x94
#define SIG_I_DATA_COBID  0x95
#define SIG_Q_HDR_COBID   0x96
#define SIG_Q_DATA_COBID  0x97

static void SpiCanTxRx(unsigned *rcv, unsigned *snd, int len) {
   unsigned *r = rcv, *s = snd, l = len, sh = 0;

   *pSPICTL = TIMOD1 | WL8 | SPIMS | SPIEN | MSBF;
   *r = 0;
   sysreg_bit_clr(sysreg_FLAGS, FLG3);
   while (l > 0) {
      *pTXSPI = (*s) >> sh;
      while ((*pSPISTAT & RXS) == 0);
      *r |= (*pRXSPI) << sh;
      l--; sh += 8;
      if (sh == 32 && l > 0) {
         s++; r++; *r = 0; sh = 0;
      }
   }
   asm("nop; nop; nop; nop; nop;");
   sysreg_bit_set(sysreg_FLAGS, FLG3);
}

static unsigned mcp_intstat(void) {
	unsigned ds = MCP_READ | (MCP_CANINTF << 8), dr;
	SpiCanTxRx(&dr, &ds, 3);
 	return (dr >> 8) & 0xFF;
}

static unsigned mcp_read_stat(void) {
	unsigned ds = MCP_READ_STAT |0xFF00, dr;
	SpiCanTxRx(&dr, &ds, 2);
 	return (dr >> 8) & 0xFF;
}

static unsigned mcp_read_err(void) {
	unsigned ds = MCP_READ | (MCP_EFLG << 8) | 0xFF0000, dr;
	SpiCanTxRx(&dr, &ds, 3);
 	return (dr >> 8) & 0xFF;
}

static void mcp_bitmod(unsigned reg, unsigned msk, unsigned data) {
	unsigned ds = MCP_BITMOD |  (reg << 8) | (msk << 16) | (data << 24), dr;
	SpiCanTxRx(&dr, &ds, 4);
}

static void mcp_read_rx(unsigned buf_nr, unsigned *data) {
   unsigned dr[4];
	data[0] = MCP_READ_RX(buf_nr);
	SpiCanTxRx(dr, data, 14);
}  

static unsigned mcp_read_rxbctrl(unsigned buf_nr) {
	unsigned ds = MCP_READ | (MCP_RXBXCTRL(buf_nr) << 8) | 0xFF0000, dr; 
	SpiCanTxRx(&dr, &ds, 3);
	return (dr >> 8) & 0xFF;
}

static void mcp_load_tx_hdr(unsigned buf, unsigned cobId, unsigned dlc, unsigned *data) {
   unsigned ds[4], dr[4];
   
   ds[0] = MCP_LOAD_TX(buf) | ((cobId & 0x7F8) << (8 - 3)) | ((cobId & 0x7) << (16 + 5));
   ds[1] = ((dlc & 0xf) << 8) | ((*data) << 16); ds[2] = ((*data) >> 16) | ((*(data + 1)) << 16);
   ds[3] = ((*(data + 1)) >> 16);
	SpiCanTxRx(dr, ds, 6 + dlc);
}

static void mcp_load_tx_data(unsigned buf, unsigned *d) {
   unsigned ds[3], dr[3];
   
   ds[0] = MCP_LOAD_TX_DATA(buf) | ((*d) << 8);
   ds[1] = ((*d) >> 24) | ((*(d + 1)) << 8);
   ds[2] = ((*(d + 1)) >> 24); 
	SpiCanTxRx(dr, ds, 9);
}

static void mcp_rts(unsigned buf_nr) {
	unsigned dr, ds; 
	ds = MCP_RTS(buf_nr);
	SpiCanTxRx(&dr, &ds, 1);
}

void mcp_restart(void) {
	unsigned ds = 0xC0, dr;
	SpiCanTxRx(&dr, &ds, 1);
	ds = MCP_READ | (MCP_CANSTAT << 8);
	SpiCanTxRx(&dr, &ds, 3);
}

void mcp_setbaud(unsigned cnf1, unsigned cnf2, unsigned cnf3) {
	unsigned ds, dr;
	
	ds = MCP_WRITE | (MCP_CNF3 << 8) | (cnf3 << 16); SpiCanTxRx(&dr, &ds, 3);
	ds = MCP_WRITE | (MCP_CNF2 << 8) | (cnf2 << 16); SpiCanTxRx(&dr, &ds, 3);
   ds = MCP_WRITE | (MCP_CNF1 << 8) | (cnf1 << 16); SpiCanTxRx(&dr, &ds, 3);
	ds = MCP_WRITE | (MCP_CANINTE << 8) | 0x3F0000; SpiCanTxRx(&dr, &ds, 3);
	ds = MCP_WRITE | (MCP_TXRTSCTRL << 8); SpiCanTxRx(&dr, &ds, 3);
	ds = MCP_WRITE | (MCP_CANCTRL << 8); SpiCanTxRx(&dr, &ds, 3);
	ds = MCP_READ | (MCP_CANSTAT << 8); SpiCanTxRx(&dr, &ds, 3);
	ds = MCP_WRITE | (MCP_RXBXCTRL(0) << 8) | 0x640000; SpiCanTxRx(&dr, &ds, 3);
	ds = MCP_WRITE | (MCP_RXBXCTRL(1) << 8) | 0x600000; SpiCanTxRx(&dr, &ds, 3);
}

static void (*RcvCallBack)(unsigned, unsigned, unsigned *); 

void initCan(void (*rcb)(unsigned cobId, unsigned dlc, unsigned *data)) {
   RcvCallBack = rcb;
   SRU(LOW, PBEN14_I);
   SRU(DAI_PB14_O, FLAG10_I);

   mcp_restart();
   mcp_setbaud(0x00, 0x99, 0x82);
}

static void doRx(unsigned stat) {
   static int ask0 = 1;
   unsigned msg[4], cobId, dlc, dat[2], buffer;
	
	if (ask0) 
	   buffer = (stat & Rx0Int_Stat) ? 0 : 1; 
	else
	   buffer = (stat & Rx1Int_Stat) ? 1 : 0;
	ask0 = !ask0;
	mcp_read_rx(buffer, msg);
	cobId = ((msg[0] & 0xFF00) >> 5) | ((msg[0] & 0xFF0000) >> 21); dlc = (msg[1] & 0xF00) >> 8;
	dat[0] = (msg[1] >> 16) | (msg[2] << 16); dat[1] = (msg[2] >> 16) | (msg[3] << 16);
	RcvCallBack(cobId, dlc, dat);
}

void pollCan(void) {
   if (!sysreg_bit_tst(sysreg_FLAGS, FLG10)) {
      unsigned stat = mcp_read_stat();
      if (stat & (Rx0Int_Stat | Rx1Int_Stat)) doRx(stat);
   }
}

static void sendData(unsigned cobId, void *d) {
   static unsigned ocb = 0xFFFFFFFF;
   if (ocb != 0xFFFFFFFF) {
      // wait for tx0 ready
      for (;;) {
         if (!sysreg_bit_tst(sysreg_FLAGS, FLG10)) {
            unsigned stat = mcp_read_stat();
            if (stat & (Rx0Int_Stat | Rx1Int_Stat)) doRx(stat);
            if (stat & Tx0Int_Stat) break;
         }
      }   
		mcp_bitmod(MCP_CANINTF, Tx0Int| WakInt | MErrInt, 0x00);
   }
   if (cobId != ocb)
      mcp_load_tx_hdr(0, cobId, 8, (unsigned *)d);
   else
      mcp_load_tx_data(0, (unsigned *)d);
   mcp_rts(0);      
   ocb = cobId;
}

void sendSpectrum(float *A, float *P, unsigned lngth, int forward, int pnts) {
   int i, j, c, s, l[2];
   float t[2];
   s = (forward) ? -1 : 1;
   l[0] = 2 * (pnts & 0xFFFFFFE); l[1] = 0;
   sendData(SPEC_A_HDR_COBID, l);
   i = (forward) ? pnts - 1 : lngth - pnts;
   for (j = 0; j < pnts; j += 2, i += 2 * s) {
      t[0] = A[i]; t[1] = A[i + s];
      sendData(SPEC_A_DATA_COBID, t);
   }
   i = (forward) ? lngth - 1 : 0;
   for (j = 0; j < pnts; j += 2, i += 2 * s) {
      t[0] = A[i]; t[1] = A[i + s];
      sendData(SPEC_A_DATA_COBID, t);
   }

   sendData(SPEC_P_HDR_COBID, l);
   i = (forward) ? pnts - 1 : lngth - pnts;
   for (j = 0; j < pnts; j += 2, i += 2 * s) {
      t[0] = P[i]; t[1] = P[i + s];
      sendData(SPEC_P_DATA_COBID, t);
   }
   i = (forward) ? lngth - 1 : 0;
   for (j = 0; j < pnts; j += 2, i += 2 * s) {
      t[0] = P[i]; t[1] = P[i + s];
      sendData(SPEC_P_DATA_COBID, t);
   }
}

void sendSignal(float *I, float *Q, unsigned lngth, int red) {
   float d = 1.0F / red, t[2];
   int i, l[2];
   l[0] = lngth / red; l[1] = 0;
   sendData(SIG_I_HDR_COBID, l);
   for (i = 0; i < lngth; i += 2 * red) {
      int j;
      float s;
      for (j = 0, s = 0.0F; j < red; j++) s += I[i + j];
      t[0] = s * d;
      for (j = red, s = 0.0F; j < 2 * red; j++) s += I[i + j];
      t[1] = s * d;
      sendData(SIG_I_DATA_COBID, t);
   }
   sendData(SIG_Q_HDR_COBID, l);
   for (i = 0; i < lngth; i += 2 * red) {
      int j;
      float s;
      for (j = 0, s = 0.0F; j < red; j++) s += Q[i + j];
      t[0] = s * d;
      for (j = red, s = 0.0F; j < 2 * red; j++) s += Q[i + j];
      t[1] = s * d;
      sendData(SIG_Q_DATA_COBID, t);
   }
}
