//-----------------------------------------------------------------------------
// File:        Canbasic.c
// Project:     CanOpen Protocol Stack
// Describtion: Access to the MCP2515 CAN chip via SPICAN.
// Date:        May 2010
// Author:      Gernot Kvas, Peter Hintenaus 
//-----------------------------------------------------------------------------
#include <string.h>
#include <linux/types.h>
#include <stdio.h>
#include <stdint.h>
#include <pthread.h>
#include <sched.h>
#include "CanBasic.h"
#include "mcp2515.h"

// The spican driver communicates with userspace with 14 byte long messages:
// the first byte contains status,
// follwed by four bytes of identifier
// follwed by a byte with the data length
// and 8 bytes of data.

static MobType MobTable[MOBTABLESZ];

static errorCallbackTy errorCallback;

static MobIdTy txmob = noMob;

static void SetupMob(uint32_t cobid, uint32_t msglen, char *dat) {
   uint8_t dlc = 0;
   uint8_t msg[14];

   msg[0] = 0;
   msg[1] = (uint8_t)(cobid >> 3); msg[2] = (uint8_t)(cobid << 5);
   msg[3] = 0; msg[4] = 0;
   msg[5] = (msglen >= 0) ? (msglen & 0xF) : (1 << 6);
   msg[6] = dat[0]; msg[7] = dat[1]; msg[8] = dat[2]; msg[9] = dat[3];
   msg[10] = dat[4]; msg[11] = dat[5]; msg[12] = dat[6]; msg[13] = dat[7];

   write(hw->can_fd, msg, 14);
}

/* Probably sends the message */
static void EnQTx(MobIdTy mobno) {
   // printf("EnQTX\n");
   take_lock();
   SetupMob(mobno->cobId, mobno->datalen, mobno->data.c);   //mob must be invalid            
   release_lock();
}

//---------------------------------------------------------------------------------
// Indexing the MOB table
//---------------------------------------------------------------------------------
void InitMobTable() {
   int i;

   for (i = 0; i < MOBTABLESZ; i++) {
      MobTable[i].mode = unused;
   }
}

//---------------------------------------------------------------------------------
// Insert:      Insert a mob to the mob table (hash table)
// Parameter:   CobIdTy cobId.........Mob id
//              modeTy mode...........Type of the mob (hard,soft,..)
// Return:      MobIdTy mobId.........Mob
//---------------------------------------------------------------------------------
MobIdTy Insert(CobIdTy cobId, modeTy mode) {
   cobId &= 0x7FF;
   MobTable[cobId].cobId = cobId;
   MobTable[cobId].mode = mode;
   return MobTable + cobId;
}

//---------------------------------------------------------------------------------
// Find:        Find a mob in the table
// Parameter:   CobIdTy cobId.........Mob id
// Return:      MobIdTy mobId.........Mob
//--------------------------------------------------------------------------------- 
static inline MobType *Find(CobIdTy cobId) {
   cobId &= 0x7FF;
   if (MobTable[cobId].mode != deleted && MobTable[cobId].mode != unused) return MobTable + cobId;
   return NULL;
} 

//---------------------------------------------------------------------------------
//  InitMob
//---------------------------------------------------------------------------------
MobIdTy InitMob(CobIdTy cobid, modeTy mode, int usrData, void (* callback)(struct MobSt *)) {
   MobIdTy mobid;

   if (cobid & 0x80000000) return noMob;
   take_lock();
   mobid = Insert(cobid, mode);
   mobid->usrData = usrData;
   mobid->nxt = noMob;
   mobid->CallBack = callback;
   release_lock();
   return mobid;
}

//------------------------------------------------------------------------------
//  DeleteMob
//------------------------------------------------------------------------------
void DeleteMob(CobIdTy cobid) {
   MobType *mob = Find(cobid);
   if (mob == NULL) return;
   take_lock();
   mob->mode = deleted;
   release_lock();
}

//-----------------------------------------------------------------------------
// set data - for a specific message object, set the data
//-----------------------------------------------------------------------------
void SetData (MobIdTy mobid, uint8_t datalen, uint8_t * data) {
   if (mobid->mode == sftrcv || mobid->mode == deleted) return;
   mobid->datalen = datalen;
   memcpy((char *)mobid->data.c, data, datalen);
}

//-----------------------------------------------------------------------------
// Send a message
//-----------------------------------------------------------------------------
void XmitMsg(MobIdTy mobid) {
   if (mobid->mode == deleted) return;
   EnQTx(mobid);
}

//-----------------------------------------------------------------------------
// send remote frame
//-----------------------------------------------------------------------------
void XmitRmtFrm(MobIdTy mobid) {
   mobid->datalen = -1; // To tell the Tx it is a Remote Frame
   EnQTx(mobid);
}

//-----------------------------------------------------------------------------
// message interrupt
//-----------------------------------------------------------------------------
void *PollCan() {
   uint8_t err;
   int intr, stat;
   static int ask0 = 1;
   uint8_t msg[14];
   MobType  *m; 
/*
   int my_policy;
   struct sched_param my_param;
   pthread_getschedparam (pthread_self (), &my_policy, &my_param);
   printf ("Can ISR running at %s/%d\n",
        (my_policy == SCHED_FIFO ? "FIFO"
        : (my_policy == SCHED_RR ? "RR"
        : (my_policy == SCHED_OTHER ? "OTHER"
        : "unknown"))),
        my_param.sched_priority);
*/
   for(;;) {
      read(hw->can_fd, msg, 14);
      take_lock();
      if (msg[0] & RX_OVR_RN) errorCallback(rxovrrn);
      if (msg[0] & WARN) errorCallback(warn);
      if (msg[0] & BUS_OFF) errorCallback(busoff);
      if (msg[0] & OK) errorCallback(ok); 

      if ((msg[0] & NO_MSG) == 0) {
         uint16_t msgid;
         if (msg[0] & REMOTE_FRAME) {
            msgid = msg[2] >> 5; msgid |= (msg[1] << 3);
            // printf("remote frame, msgid = 0x%x\n", msgid);
            m = Find(msgid);
            // printf("remote frame, m = 0x%x\n", m);
            if (m != NULL && m->mode == hrdxmit) EnQTx(m);
         } else if (msg[0] & TX_ACK) {
            msgid = msg[2] >> 5; msgid |= (msg[1] << 3);
            m = Find(msgid);
            if (m != NULL) m->CallBack(m);
         } else {
            msgid = msg[2] >> 5; msgid |= (msg[1] << 3);            
            m = Find(msgid);
            if (m != NULL) {
               m->datalen = msg[5];
               m->data.c[0] = msg[6]; m->data.c[1] = msg[7]; m->data.c[2] = msg[8]; m->data.c[3] = msg[9];
               m->data.c[4] = msg[10]; m->data.c[5] = msg[11]; m->data.c[6] = msg[12]; m->data.c[7] = msg[13];
               m->CallBack(m);                     // Callback for this id
            }
         }        
      }
      release_lock();
   }
   return NULL;
}

static uint32_t baudrate;

void InitCan(uint32_t b, errorCallbackTy ecb) {
   char t0, t1;

   errorCallback = ecb;
   InitMobTable();

   mcp_init("/dev/spican0");
   mcp_reset();
   mcp_setbaud(baudrate = b);
   mcp_start(PollCan);

   // Do some chip setup

}   

void ChkNRestartChip() {
   mcp_reset();
   mcp_setbaud(baudrate);
}

void ChipOff() {
   mcp_reset();
}

