#include <stdio.h>
#include <pthread.h>
#include <signal.h>
#include <time.h>
#include <sched.h>
#include <limits.h>

#include "lock.h"
#include "Event.h"
#include "Priorities.h"

static pthread_attr_t timer_attrs;
static struct sched_param sp;   

void initEvents(void) {
   int iret;

   sp.sched_priority = CANOPENTIMERP;
   iret = pthread_attr_init(&timer_attrs);
   iret = pthread_attr_setschedpolicy(&timer_attrs, SCHED_FIFO);   
   iret = pthread_attr_setschedparam(&timer_attrs, &sp);
   iret = pthread_attr_setinheritsched (&timer_attrs, PTHREAD_EXPLICIT_SCHED);
   iret = pthread_attr_setdetachstate(&timer_attrs, PTHREAD_CREATE_DETACHED);
   iret = pthread_attr_setstacksize(&timer_attrs, (0x20000 > PTHREAD_STACK_MIN) ? 0x20000 : PTHREAD_STACK_MIN);
}

void SetupEvent(EventTy *e, void (*action)(union sigval p), int par) {
   take_lock();
   if (!e->inited) {
      e->evp.sigev_notify = SIGEV_THREAD;
      e->evp.sigev_notify_attributes = &timer_attrs;
      e->evp.sigev_value.sival_int = par;
      e->evp.sigev_notify_function = action;
//      if (timer_create(CLOCK_MONOTONIC, &(e->evp), &(e->tid))) {
//         perror("SetupEvent, timer_create");
//      }
//      e->inited = 1;
   }
   release_lock();
}

// the unit of the delay is 1/f s
void reSchedule(EventTy *event, uint32_t delay, uint32_t f) {
   int32_t sec = delay / f, nsec = (delay % f) * (1000000000 / f);
   struct itimerspec tv = {0, 0, sec, nsec};

   take_lock();
   if (!event->inited) {
      if (timer_create(CLOCK_MONOTONIC, &(event->evp), &(event->tid))) {
         perror("reSchedule, timer_create");
      }
      event->inited = 1;
   }
   if (timer_settime(event->tid, 0, &tv, NULL)) {
      perror("reSchedule, timer_settime");
   }
   release_lock();
}

void schedulePeriodic(EventTy *event, uint32_t delay, uint32_t period, uint32_t f) {
   int32_t dsec = delay / f, dnsec = (delay % f) * (1000000000 / f);
   int32_t psec = period / f, pnsec = (period % f) * (1000000000 / f);
   struct itimerspec tv = {psec, pnsec, dsec, dnsec};
   
   // printf("schedulePeriodic: psec = %d, pnsec = %d\n", psec, pnsec);
   take_lock();
   if (!event->inited) {
      if (timer_create(CLOCK_MONOTONIC, &(event->evp), &(event->tid))) {
         perror("reSchedule, timer_create");
      }
      event->inited = 1;
   }
   if (timer_settime(event->tid, 0, &tv, NULL)) {
      perror("schedulePeriodic, timer_settime");
   }
   release_lock();
}

void cancel(EventTy *event) {
   struct itimerspec tv = {0, 0, 0, 0};
   take_lock();
   if (event->inited) {
      if (timer_settime(event->tid, 0, &tv, NULL)) {
         perror("cancel, timer_settime");
      }
   }
   release_lock();
}

