#include <signal.h>
#include <pthread.h>
#include <time.h>
#include <stdio.h>
#include <stdint.h>
#include "../vm/vm.h"
#include "error.h"
#include "lock.h"
#include "size.h"
#include "nmt.h"
#include "ComObj.h"

uint8_t NmtId;
NmtStateTy NmtState;

uint16_t HBtime;
uint16_t GrdTime;
uint8_t lifetime;
static uint8_t GrdToggle;

static EventTy HbEvent, GrdEvent;

static MobIdTy Nmt0, ECmob;
static int Nmt0Locked = 0;
static pthread_mutex_t Nmt0Mutex;
static pthread_cond_t Nmt0Cond;

// Object dictionary entries for DSP302
// DSP 302
uint32_t NMTStartup, BootTime;
uint8_t *SlaveOd[127];
uint32_t ExpAppSwDate[127], ExpAppSwTime[127], SlaveAssignment[127];
uint32_t DevTypeId[127], Entry1F85_1F88[4][127];

static pthread_t Booter[127];
static int all_booted = 0;
static pthread_mutex_t bootMutex;
static pthread_cond_t bootCond;
pthread_attr_t nmtAttrs;

#define ISSLAVE       0x01L
#define ERRORCTLSTRT  0x02L
#define AUTOSTRT      0x04L
#define MANDATRYSLAVE 0x08L
#define KEEPALIVE     0x10L
#define VERIFYAPPLSW  0x20L
#define APPLSWDWNLD   0x40L

NmtStateTy GetRemoteNodeState(uint8_t nodeId) {
   RemoteNodeTy *n = &(RemoteNode[nodeId - 1]);
   return n->state;
}

static void SetUsrBit(uint8_t nodeId, int v) {
   RemoteNodeTy *n = &(RemoteNode[nodeId - 1]);
   if (v) n->errorctrl |= usrmsk;
   else n->errorctrl &= ~usrmsk;
}

static int GetUsrBit(uint8_t nodeId) {
   RemoteNodeTy *n = &(RemoteNode[nodeId - 1]);
   return n->errorctrl & usrmsk;
}

void ResetBootedBit(uint8_t nodeId) {
   RemoteNodeTy *n = &(RemoteNode[nodeId - 1]);
   n->errorctrl &= ~bootedmsk;
}

int GetBootedBit(uint8_t nodeId) {
   RemoteNodeTy *n = &(RemoteNode[nodeId - 1]);
   return n->errorctrl & bootedmsk;
}

void NmtSent(MobType  *foo) {
   pthread_mutex_lock(&Nmt0Mutex);
   Nmt0Locked = 0;
   pthread_cond_signal(&Nmt0Cond);
   pthread_mutex_unlock(&Nmt0Mutex);
}

static void (*StateCng)(NmtStateTy, StateCngCmdTy);

void NmtRcvd(MobType  *foo) {
   uint8_t d[1];
   NmtStateTy old = NmtState;
   StateCngCmdTy cmd;
   
   if (Nmt0->data.c[1] != 0 && Nmt0->data.c[1] != NmtId) return;

   if (Nmt0->data.c[0] == 1) {
      NmtState = operational; cmd = keepWrkng;
   } else if (Nmt0->data.c[0] == 2) {
      NmtState = stopped; cmd = keepWrkng; 
   } else if (Nmt0->data.c[0] == 128) {
      NmtState = preop; cmd = enterPreop;
   } else if (Nmt0->data.c[0] == 129) {
      NmtState = boot; cmd = resetAppl;
   } else if (Nmt0->data.c[0] == 130) {
      NmtState = boot; cmd = resetCom;
   } else return;

   if (HBtime == 0 && (NmtState == preop || NmtState == stopped || NmtState == operational)) { 
      d[0] = GrdToggle | (uint8_t)NmtState;
      SetData(ECmob, 1, d);
   }

   StateCng(old, cmd);
}

static void HBresend(union sigval d) {
   // printf("HBresend\n");
   take_lock();
   SetData(ECmob, 1, (uint8_t *)&NmtState);
   XmitMsg(ECmob);
   release_lock();
}

vm *LifeTimeTask = NULL;
int LTindSent;

static void startLifeTimeTask(int occured) {
   if (LifeTimeTask != NULL) {
      if (LifeTimeTask->running) {
         char reason[100];
         sprintf(reason, "realtime violation in task %20s", LifeTimeTask->name);
         logError(0xff, reason);
      } else {
         LifeTimeTask->statics[0] = occured;
         stdWakeup(LifeTimeTask);
      }
   }
}

static void LifeTimeInd() {
   take_lock();
   startLifeTimeTask(1);
   LTindSent = 1;
   release_lock();
}

void ECSent() {
   if (NmtState == boot) {
      SetData(ECmob, 1, (uint8_t *)&NmtState);	// its little endian
      return;
   }
   if (HBtime == 0) {
      uint8_t d[1];
      GrdToggle ^= 0x80;
      d[0] = GrdToggle | (uint8_t)NmtState;
      SetData(ECmob, 1, d);
      if (GrdTime != 0 && lifetime != 0) reSchedule(&GrdEvent, lifetime * GrdTime, 1000);
      if (LTindSent) {
         startLifeTimeTask(0);
         LTindSent = 0;
      }
   }
}

void StopErrorControl() {
   take_lock();
   if (HBtime != 0) cancel(&HbEvent);
   else if (GrdTime != 0 && lifetime != 0) cancel(&GrdEvent);
   DeleteMob(ECmob->cobId);
   release_lock();
}

RemoteNodeTy RemoteNode[127];
ConsumerHBtimeTy ConsumerHBtime[127];
static int nmtMaster;

void initNmtEvents(void) {
   int i;
   for(i = 0; i < 127; i++) RemoteNode[i].event.inited = 0;
   GrdEvent.inited = 0;
   HbEvent.inited = 0;
}


void InitNmt(int mstr, void (*Cng)(NmtStateTy, StateCngCmdTy)) {
   int i;

   take_lock();
   NmtState   = preop;
   StateCng = Cng;
   nmtMaster = mstr;
   if (nmtMaster) Nmt0 = InitMob(0x0, sftxmit, 0, NmtSent);
   else Nmt0 = InitMob(0x0, sftrcv, 0, NmtRcvd);
   SetupEvent(&GrdEvent, LifeTimeInd, 0);
   SetupEvent(&HbEvent, HBresend, 0); 
   ECmob = InitMob((CobIdTy)(1792 + NmtId), hrdxmit, 0, ECSent);
   for (i = 0; i < 127; i++) {
      RemoteNode[i].state = unknown;
      RemoteNode[i].errorctrlMob = noMob;
      RemoteNode[i].errorctrl = 0;
      RemoteNode[i].inBoot = 0;
      SetupEvent(&RemoteNode[i].event, NULL, 0);
   }
   Nmt0Locked = 0;
   if (pthread_mutex_init(&Nmt0Mutex, NULL)) fprintf(stderr, "InitNmt, pthread_mutex_init");
   if (pthread_cond_init(&Nmt0Cond, NULL)) fprintf(stderr, "InitNmt pthread_cond_init");

   release_lock();
}

int IsNmtMaster() {return nmtMaster;}

void Bootup() {
   uint8_t d = 0;
   SetData(ECmob, 1, &d);
   XmitMsg(ECmob);
}

void StartHeartBeat() {
   schedulePeriodic(&HbEvent, HBtime - 1, HBtime, 1000);      
}

void SetupGuarding(void) {
   GrdToggle = 0;
   SetData(ECmob, 1, (uint8_t *)&NmtState);
}

// master procedures

void StartRemoteNode(uint8_t nodeId) {
   uint8_t d[2];
   NmtStateTy old = NmtState;
   if (nodeId == NmtId || nodeId == 0) {
      NmtState = operational;
      StateCng(old, keepWrkng);
      if (nodeId != 0) return;
   }
   pthread_mutex_lock(&Nmt0Mutex);
   while(Nmt0Locked) pthread_cond_wait(&Nmt0Cond, &Nmt0Mutex);
   Nmt0Locked = 1;
   pthread_mutex_unlock(&Nmt0Mutex);
   d[0] = 1; d[1] = nodeId;
   SetData(Nmt0, 2, d);
   take_lock();
   if (nodeId == 0 || RemoteNode[nodeId-1].state != operational) XmitMsg(Nmt0);
   if (nodeId != 0) {
      RemoteNode[nodeId-1].errorctrl |= statechngdmsk;
      RemoteNode[nodeId-1].state = operational;
   } else { 
      int i;
      for (i = 0; i < 127; i++) if (RemoteNode[i].errorctrlMob != noMob) {
         RemoteNode[i].errorctrl |= statechngdmsk;
         RemoteNode[i].state = operational;
      }
   }
   release_lock();
}

void StopRemoteNode(uint8_t nodeId) {
   uint8_t d[2];
   NmtStateTy old = NmtState;
   if (nodeId == NmtId || nodeId == 0) {
      NmtState = stopped;
      StateCng(old, keepWrkng);
      if (nodeId != 0) return;
   }
   pthread_mutex_lock(&Nmt0Mutex);
   while(Nmt0Locked) pthread_cond_wait(&Nmt0Cond, &Nmt0Mutex);
   Nmt0Locked = 1;
   pthread_mutex_unlock(&Nmt0Mutex);
   d[0] = 2; d[1] = nodeId;
   SetData(Nmt0, 2, d);
   take_lock();
   XmitMsg(Nmt0);
   if (nodeId != 0) {
      RemoteNode[nodeId-1].errorctrl |= statechngdmsk;
      RemoteNode[nodeId-1].state = stopped;
   } else {
      int i;
      for (i = 0; i < 127; i++) if (RemoteNode[i].errorctrlMob != noMob) {
         RemoteNode[i].errorctrl |= statechngdmsk;
         RemoteNode[i].state = stopped;
      }
   }
   release_lock();
}

void EnterPreOp(uint8_t nodeId) {
   uint8_t d[2];
   NmtStateTy old = NmtState;
   if (nodeId == NmtId || nodeId == 0) {
      NmtState = preop;
      StateCng(old, enterPreop);
      if (nodeId != 0) return;
   }
   pthread_mutex_lock(&Nmt0Mutex);
   while(Nmt0Locked) pthread_cond_wait(&Nmt0Cond, &Nmt0Mutex);
   Nmt0Locked = 1;
   pthread_mutex_unlock(&Nmt0Mutex);
   d[0] = 128; d[1] = nodeId;
   SetData(Nmt0, 2, d);
   take_lock();
   XmitMsg(Nmt0);
   if (nodeId != 0) {
      RemoteNode[nodeId-1].errorctrl |= statechngdmsk;
      RemoteNode[nodeId-1].state = preop;
   } else {
      int i;
      for (i = 0; i < 127; i++) if (RemoteNode[i].errorctrlMob != noMob) {
         RemoteNode[i].errorctrl |= statechngdmsk;
         RemoteNode[i].state = preop;
      }
   }
   release_lock();
}

void ResetNode(uint8_t nodeId) {
   uint8_t d[2];
   NmtStateTy old = NmtState;
   if (nodeId == NmtId) {
      NmtState = boot;
      StateCng(old, resetAppl); return;
   }
   if (nodeId == 0) {
      NmtState = boot;
      StateCng(old, resetApplNStp);
   }
   pthread_mutex_lock(&Nmt0Mutex);
   while(Nmt0Locked) pthread_cond_wait(&Nmt0Cond, &Nmt0Mutex);
   Nmt0Locked = 1;
   pthread_mutex_unlock(&Nmt0Mutex);
   d[0] = 129; d[1] = nodeId;
   SetData(Nmt0, 2, d);
   take_lock();
   XmitMsg(Nmt0);
   if (nodeId != 0) {
      RemoteNode[nodeId - 1].errorctrl &= (~errorctrlon);   // stop error control
      RemoteNode[nodeId-1].errorctrl |= statechngdmsk;
      RemoteNode[nodeId-1].state = boot;
      cancel(&RemoteNode[nodeId-1].event);
      release_lock();
   } else {
      int i;
      for (i = 0; i < 127; i++) if (RemoteNode[i].errorctrlMob != noMob) {
         RemoteNode[i].errorctrl &= (~errorctrlon);
         RemoteNode[i].errorctrl |= statechngdmsk;
         RemoteNode[i].state = boot;
         cancel(&RemoteNode[i].event);
      }
      release_lock();
      StateCng(boot, resetCom);
   }
}

void ResetCom(uint8_t nodeId) {
   uint8_t d[2];
   NmtStateTy old = NmtState;
   if (nodeId == NmtId || nodeId == 0) {
      NmtState = boot;
      StateCng(old, resetCom);
      if (nodeId != 0) return;
   }
   pthread_mutex_lock(&Nmt0Mutex);
   while(Nmt0Locked) pthread_cond_wait(&Nmt0Cond, &Nmt0Mutex);
   Nmt0Locked = 1;
   pthread_mutex_unlock(&Nmt0Mutex);
   d[0] = 130; d[1] = nodeId;
   SetData(Nmt0, 2, d);
   take_lock();
   XmitMsg(Nmt0);
   if (nodeId != 0) {
      RemoteNode[nodeId - 1].errorctrl &= (~errorctrlon);   // stop error control
      RemoteNode[nodeId-1].errorctrl |= statechngdmsk;
      RemoteNode[nodeId-1].state = boot;
      cancel(&RemoteNode[nodeId-1].event);
   } else {
      int i;
      for (i = 0; i < 127; i++) if (RemoteNode[i].errorctrlMob != noMob) {
         RemoteNode[i].errorctrl &= (~errorctrlon);
         RemoteNode[i].errorctrl |= statechngdmsk;
         RemoteNode[i].state = boot;
         cancel(&RemoteNode[i].event);
      }
   }
   release_lock();
}

vm *RemoteErrorControlTask = NULL;

static void startEcTask(uint8_t nodeid, int32_t occured, int32_t reason) {
   char err[100];
   sprintf(err, "CANopen: errorcontrol for node %d %s, reason = %x", nodeid, (occured) ? "occured" : "cleared", reason);
   logError(0xfd, err);
   if (RemoteErrorControlTask != NULL) {
      if (RemoteErrorControlTask->running) {
         sprintf(err, "realtime violation in task %20s", RemoteErrorControlTask->name);
         logError(0xff, err);
      } else {
         RemoteErrorControlTask->statics[0] = nodeid;
         RemoteErrorControlTask->statics[1] = occured;
         RemoteErrorControlTask->statics[2] = reason;
         stdWakeup(RemoteErrorControlTask);
      }
   }
}

static int mandatory = 0, optional = 0, mandbooted = 0, optbooted = 0;

vm *bootedTask = NULL;

static void NodeBooted(uint8_t nodeid) {
   char reason[100];
   pthread_mutex_lock(&bootMutex);
   if (SlaveAssignment[nodeid-1] & MANDATRYSLAVE) {
      mandbooted++;
      pthread_cond_signal(&bootCond);
   } else optbooted++;
   pthread_mutex_unlock(&bootMutex);
   sprintf(reason, "CANopen: node %x booted", nodeid); 
   logError(0xfd, reason);
   if (bootedTask != NULL) {
      if (bootedTask->running) {
         sprintf(reason, "realtime violation in task %20s", bootedTask->name);
         logError(0xff, reason);
      } else {
         bootedTask->statics[0] = nodeid;
         stdWakeup(bootedTask);
      }
   }  
   if (NmtState == operational || !(NMTStartup & 0x2)) StartRemoteNode(nodeid);
}

static void *BootSlave(void *s);

static void ResetNReboot(uint8_t nodeid) {
   ResetCom(nodeid);
   if (!RemoteNode[nodeid - 1].inBoot) {
      if (pthread_create(Booter + nodeid - 1, &nmtAttrs, BootSlave, RemoteNode[nodeid - 1].clSdo)) {
         fprintf(stderr, "ResetNReboot, pthread_create Booter");
      }
      pthread_detach(Booter[nodeid - 1]);
   }
}

static void report(uint8_t nodeid, NmtStateTy newst, int32_t err, int32_t reason) {
   RemoteNodeTy *n = &(RemoteNode[nodeid - 1]);
   if (IsNmtMaster() && n->inBoot) {   
      // this is a keep alive node, or one with error control just started
      if ((newst & 0x7f) == operational) {
         n->state = operational;
         n->errorctrl |= ((newst ^ togglemsk) & togglemsk);   // copy toggle bit (for node guarding)
         n->errorctrl |= bootedmsk;
         startEcTask(nodeid, 1, BOOTERRORL);   // it's a warning actually
         n->inBoot = 0;
         NodeBooted(nodeid);
      } else if ((newst & 0x7f) == unknown) {   // error control timed out   
         startEcTask(nodeid, 1, (n->errorctrl & guardingmsk) ? BOOTERRORF : BOOTERRORE);
         n->inBoot = 0;
      } else if ((newst & 0x7f) == preop) {
         n->inBoot = 0;
         n->state = preop;
         n->errorctrl |= bootedmsk;
         NodeBooted(nodeid);         // node is ready for start!
      } else { // node is not operational
         ResetCom(nodeid);
      } 
   } else { // normal error indication
      int savenode = 0;
      if (err) {
         if (n->errorctrl & errorindmsk) { // two indications in a row, there is something wrong!
            startEcTask(nodeid, 1, reason);
            ResetBootedBit(nodeid);
            if (!GetUsrBit(nodeid)) {
               ReportEmcy(0x8130);
               if (IsNmtMaster()) {
                  if (SlaveAssignment[nodeid - 1] & MANDATRYSLAVE) {
                     pthread_mutex_lock(&bootMutex);
                     mandbooted--;
                     pthread_cond_signal(&bootCond);
                     pthread_mutex_unlock(&bootMutex);
                     if (NMTStartup & 0x10) ResetNode(0);
                       else savenode = 1;
                  } else {
                     pthread_mutex_lock(&bootMutex);
                     optbooted--;
                     pthread_mutex_unlock(&bootMutex);
                     savenode = 1;
                    }
               }
            }
            SetUsrBit(nodeid, 1);
            if (savenode) ResetNReboot(nodeid);
         }
      } else {
         if (GetUsrBit(nodeid)) ClearEmcy(0x8130);
         SetUsrBit(nodeid, 0);
         startEcTask(nodeid, 0, 0);
      }         
   }
}

static void GuardTimer(union sigval ni) {
   RemoteNodeTy *n = &(RemoteNode[ni.sival_int]);
   uint32_t st = unknown;
   uint8_t retry;
   uint32_t guardtime;

   take_lock();
   if (!(n->errorctrl & errorctrlon)) {
      release_lock();
      return;
   }
   GetGuardingParams(ni.sival_int + 1, &retry, &guardtime);
   if (!(n->errorctrl & guardrcvd)) {
      if (n->guardretry >= retry) { 
         report(ni.sival_int + 1, st, 1, 0);
         n->errorctrl |= errorindmsk;
      } else n->guardretry++;
   } else {
      n->guardretry = 0;
       if (((st = n->errorctrlMob->data.c[0]) & togglemsk) != (n->errorctrl & togglemsk) |
         (!(n->errorctrl & statechngdmsk) && (st & (~togglemsk)) != n->state)) { 
         n->errorctrl &= (~statechngdmsk);
         report(ni.sival_int + 1, st, 1, 1);
         n->errorctrl |= errorindmsk;
      } else if ((n->errorctrl & errorindmsk) != 0) {
         n->errorctrl &= (~statechngdmsk);
         report(ni.sival_int + 1, st, 0, 0);
         n->errorctrl &= (~errorindmsk);
      } else n->errorctrl &= (~statechngdmsk);
   }
   if ((n->errorctrlMob->data.c[0] & 0x80) == (n->errorctrl & 0x80)) n->errorctrl ^= 0x80;
   n->errorctrl &= (~guardrcvd);
   XmitRmtFrm(n->errorctrlMob);
   reSchedule(&(n->event), guardtime, 1000);
   release_lock();
}

static void GuardRcvd(MobType  *mob) {
   int nodeidx = mob->usrData;
   RemoteNode[nodeidx].errorctrl |= guardrcvd;
}

void StartGuarding(uint8_t nodeid) {
   RemoteNodeTy *n = &(RemoteNode[nodeid - 1]);
   uint8_t retry;
   uint32_t guardtime;

   GetGuardingParams(nodeid, &retry, &guardtime);
   n->errorctrl &= usrmsk;      // user bit must survive reboot!!!
   n->errorctrl |= guardingmsk | errorctrlon;
   n->guardretry = 0;
   if (n->errorctrlMob == noMob) n->errorctrlMob = InitMob(1792 + nodeid, sftrcv, nodeid - 1, GuardRcvd); 
   SetupEvent(&(n->event), GuardTimer, nodeid - 1);
   XmitRmtFrm(n->errorctrlMob);
   reSchedule(&(n->event), guardtime, 1000);
}

static void HeartBeatTimeout(union sigval ni) {
   RemoteNodeTy *n = &(RemoteNode[ni.sival_int]);

   take_lock();
   if (!(n->errorctrl & errorctrlon)) {
      release_lock();
      return;
   }
   report(ni.sival_int + 1, unknown, 1, 0);
   n->errorctrl |= errorindmsk;
   reSchedule(&(n->event), ConsumerHBtime[n->CHBtimeIdx].HBtime, 1000);
   release_lock();
}

static void HeartBeatCallback(MobType  *m) {
   RemoteNodeTy *n = RemoteNode + m->usrData;
   uint32_t st;

   // printf("Heartbeat received\n");
   if (!(n->errorctrl & errorctrlon)) return;
   reSchedule(&(n->event), ConsumerHBtime[n->CHBtimeIdx].HBtime, 1000);
   if (nmtMaster && !(n->errorctrl & statechngdmsk) && (st = m->data.c[0]) != n->state) {
      n->errorctrl &= (~statechngdmsk);
      report(m->usrData + 1, st, 1, 1);
      n->errorctrl |= errorindmsk;
   } else if (n->errorctrl & errorindmsk) {
      n->errorctrl &= (~statechngdmsk);
      report(m->usrData + 1, st, 0, 0);
      n->errorctrl &= (~errorindmsk);
   } else n->errorctrl &= (~statechngdmsk);
   // printf("Heartbeat done\n");
}

void ConsumeHeartBeat(ConsumerHBtimeTy *CHBtime) {
   RemoteNodeTy *n = &(RemoteNode[CHBtime->id - 1]);

   n->errorctrl &= usrmsk;      // user bit must survive reboot!!!
   n->errorctrl |= errorctrlon;
   if (!nmtMaster) n->errorctrl |= errorindmsk; // force an indication with not occured at the beginning
   if (n->errorctrlMob == noMob) 
      n->errorctrlMob = InitMob(1792 + CHBtime->id, sftrcv, CHBtime->id - 1, HeartBeatCallback); 
   SetupEvent(&(n->event), HeartBeatTimeout, CHBtime->id - 1);
   reSchedule(&(n->event), CHBtime->HBtime, 1000);
}

void StopECConsumer() {
   int i;
   for (i = 0; i < 127; i++) {
      DeleteMob(RemoteNode[i].errorctrlMob->cobId);
      RemoteNode[i].errorctrl &= ~errorctrlon;
   }
}

// Network boot according to DSP 302 V3.0

static int check1F85_1F88(SdoTy *sdo);
static int CheckNodeState(SdoTy *sdo, int start);
static int checkApplSW(SdoTy *sdo);
static int configure(SdoTy *sdo);

static int doBootSlave(SdoTy *sdo) {
   uint32_t resp, result;
   uint32_t expDevId;

   result = SdoUpload(sdo, (char *) &resp, 4, 0x1000, 0);
   if (result != 0) {       // SDO timeout or other problem
      startEcTask(sdo->nodeId, 1, BOOTERRORB);
      return 0;
   }
   // fprintf(stderr, "CANopen, booting node %d\n", sdo->nodeId);
   expDevId = DevTypeId[sdo->nodeId - 1];
   if ( expDevId != 0 && resp != expDevId) {// 0x1f84
      startEcTask(sdo->nodeId, 1, BOOTERRORC);
      return 0;
   }
   if (!check1F85_1F88(sdo)) return 0;
   if (CheckNodeState(sdo, 0)) return 1; // give keep alive node a chance...

   if (!checkApplSW(sdo)) return 0;
   if (!configure(sdo)) return 0;
   return CheckNodeState(sdo, 1);
}

static int check1F85_1F88(SdoTy *sdo) {
   uint32_t resp, result, cnt;
   uint32_t entry;

   cnt = 0;
   do {
      entry = Entry1F85_1F88[cnt][sdo->nodeId - 1];
      if (entry != 0) {
         result = SdoUpload(sdo, (char *) &resp, 4, 0x1018, cnt + 1);
         if (result != 0) {
            if (result != SDOTIMEOUTERR) RemoteNode[sdo->nodeId - 1].state = dead;
            startEcTask(sdo->nodeId, 1, BOOTERRORD + cnt);
            return 0;
         }
         if (entry != resp) {
            startEcTask(sdo->nodeId, 1, BOOTERRORD + cnt);
            return 0;
         }
      }
      cnt++;
   } while (cnt < 4);
   return 1;
}

static int CheckNodeState(SdoTy *sdo, int start) {
   if ((SlaveAssignment[sdo->nodeId - 1] & KEEPALIVE) || start) {
      RemoteNodeTy *n = &RemoteNode[sdo->nodeId-1];
//    n->inBoot = !start;
      n->state = unknown;      // force an indication on first error control response 
      n->errorctrl &= (~statechngdmsk);
      // fast check for object 0x1016
      if (n->CHBtimeIdx == 128 || ConsumerHBtime[n->CHBtimeIdx].HBtime == 0) {
         if ((SlaveAssignment[sdo->nodeId - 1] & 0xffffff00) != 0)
            StartGuarding(sdo->nodeId);
         else if (start) {
            n->inBoot = 0;
            n->state = preop;
            NodeBooted(sdo->nodeId);
         }
      } else 
         ConsumeHeartBeat(&ConsumerHBtime[n->CHBtimeIdx]);
      return 1;
   }
   return 0;
}

static int checkApplSW(SdoTy *sdo) {
   uint32_t resp, result, entry;
   if (SlaveAssignment[sdo->nodeId] & VERIFYAPPLSW) {
      result = SdoUpload(sdo, (char *) &resp, 4, 0x1f52, 1);
      if (result != 0) {
         if (result != SDOTIMEOUTERR) RemoteNode[sdo->nodeId - 1].state = dead;         
         startEcTask(sdo->nodeId, 1, BOOTERRORH);
         return 0;
      }
      entry = ExpAppSwDate[sdo->nodeId];
      if (entry == 0) { 
         startEcTask(sdo->nodeId, 1, BOOTERRORG);
         RemoteNode[sdo->nodeId - 1].state = dead;
         return 0;
      } else if (entry < resp) {
         startEcTask(sdo->nodeId, 1, BOOTERRORH);
         RemoteNode[sdo->nodeId - 1].state = dead;
         return 0;
      } else if (entry == resp) {
         result = SdoUpload(sdo, (char *) &resp, 4, 0x1f52, 2);
         if (result != 0) {
            if (result != SDOTIMEOUTERR) RemoteNode[sdo->nodeId - 1].state = dead;         
            startEcTask(sdo->nodeId, 1, BOOTERRORH);
            return 0;
         }
         entry = ExpAppSwTime[sdo->nodeId - 1];
         if (entry == 0) {
            startEcTask(sdo->nodeId, 1, BOOTERRORG);
            RemoteNode[sdo->nodeId - 1].state = dead;
            return 0;
         } else if (entry < resp) {
            startEcTask(sdo->nodeId, 1, BOOTERRORH);
            RemoteNode[sdo->nodeId - 1].state = dead;
            return 0;
         }
      }
   }
   return 1;
}

static int configure(SdoTy *sdo) {
   uint8_t *od = SlaveOd[sdo->nodeId - 1];
   uint32_t len, result;
   struct timespec st = {.tv_sec = 0, .tv_nsec = 100000000};

   // ResetCom(sdo->nodeId);      // makes sure that node guarding restarts with defined toggle bit. Beckhoff does not like it!!!
   EnterPreOp(sdo->nodeId);
   if (od == NULL) len = 0;
   else {
      len = ((uint32_t)*od) | (((uint32_t)*(od + 1)) << 8) | (((uint32_t)*(od + 2)) << 16) | (((uint32_t)*(od + 3)) << 24);
      od += 4;
   }

   nanosleep(&st, NULL);                        // Möller fix. Möller devices don't listen
                                                // for 20 mS after a ResetCommunication
   while (len > 0) {
      uint16_t idx = (uint16_t)(od[0]) | ((uint16_t)(od[1]) << 8);
      uint8_t subidx = od[2];
      uint32_t size, resp;

      // fprintf(stderr, "configuring idx = %x subidx = %x ", idx, subidx);

      if (idx == 0x1005 || idx == 0x1012|| idx == 0x1014 ||      // Sync, Time, Emcy
         (idx == 0x1028 && subidx > 0) ||                        // consumed Emcy    
         (0x1400 <= idx && idx <= 0x15FF && subidx == 1) ||      // receive PDO's
         (0x1800 <= idx && idx <= 0x19FF && subidx == 1))      // transmit PDO's
      {
         result = SdoUpload(sdo, (char *) &resp, 4, idx, subidx);
         if (result != 0) {
            if (result != SDOTIMEOUTERR) RemoteNode[sdo->nodeId - 1].state = dead;         
            startEcTask(sdo->nodeId, 1, BOOTERRORJ);
            return 0;
         }
         if (idx == 0x1005) resp &= 0xBFFFFFFF;
         else if (idx == 0x1012) resp &= 0x3FFFFFFF;
         else resp |= 0x80000000;   
         result = SdoDwnLoad(sdo, (char *)&resp, 4, idx, subidx);
         if (result != 0) {
            if (result != SDOTIMEOUTERR) RemoteNode[sdo->nodeId - 1].state = dead;         
            startEcTask(sdo->nodeId, 1, BOOTERRORJ);
            return 0;
         }
         od += 3;
         size = (uint32_t)(od[0]) | ((uint32_t)(od[1]) << 8) | ((uint32_t)(od[2]) << 16) | ((uint32_t)(od[3]) << 24);
         od += 4;
         // fprintf(stderr, "size = %x\n", size);
         result = SdoDwnLoad(sdo, od, (int)size, idx, subidx);
         if (result != 0) {
            if (result != SDOTIMEOUTERR) RemoteNode[sdo->nodeId - 1].state = dead;         
            startEcTask(sdo->nodeId, 1, BOOTERRORJ);
            return 0;
         }
         od += size;
         len--;
      } else {          
         od += 3;
         size = (uint32_t)(od[0]) | ((uint32_t)(od[1]) << 8) | ((uint32_t)(od[2]) << 16) | ((uint32_t)(od[3]) << 24);
         od += 4;
         // fprintf(stderr, "size = %x\n", size);
         result = SdoDwnLoad(sdo, od, size, idx, subidx);
         if (result != 0) {
            if (result != SDOTIMEOUTERR) RemoteNode[sdo->nodeId - 1].state = dead;         
            startEcTask(sdo->nodeId, 1, BOOTERRORJ);
            return 0;
         } 
         od += size;
         len--;
      }
   }
   return 1;
}
static void *BootSlave(void *s) {
   SdoTy *sdo = (SdoTy *)s;
   struct timespec st = {.tv_sec = 1, .tv_nsec = 0};

   // printf("in BootSlave, booting %d\n", sdo->nodeId);
   for(;;) {
      RemoteNode[sdo->nodeId - 1].inBoot = 1;
      if (doBootSlave(sdo)) break;
      if (RemoteNode[sdo->nodeId - 1].state == dead) {
         RemoteNode[sdo->nodeId - 1].state == unknown;
         // not so good, BoschRexroth drives sometimes ignore the enterpreop when they
         // get configured and abort SDO downloads thereafter. Another try fixes this.
//       break;   
      }
      nanosleep(&st, NULL);
   }
   return NULL;
}

void BootNetwork(void) {
   int i, ni;

   if (pthread_mutex_init(&bootMutex, NULL)) fprintf(stderr, "BootNetwork, pthread_mutex_init");
   if (pthread_cond_init(&bootCond, NULL))   fprintf(stderr, "BootNetwork, pthread_cond_init");
   mandatory = mandbooted = optional = optbooted = 0;

   for (i = 0; i < CSDONO; i++) {
      if ((ClientSdo[i].Client2Server & 0x80000000) || (ClientSdo[i].Server2Client & 0x80000000) ||
         (ni = ClientSdo[i].nodeId) == 0 || (!(SlaveAssignment[ni-1] & ISSLAVE))) continue;
      if (SlaveAssignment[ni-1] & MANDATRYSLAVE) mandatory++; 
      else optional++;

      SetupClientSdo(i);
      if (pthread_create(Booter + ni - 1, &nmtAttrs, BootSlave, ClientSdo + i)) {
         fprintf(stderr, "BootNetwork, pthread_create Booter");
      }
      pthread_detach(Booter[ni - 1]);
   }
   if (mandatory != 0) {
      pthread_mutex_lock(&bootMutex);
      while (mandatory != mandbooted) pthread_cond_wait(&bootCond, &bootMutex);
      pthread_mutex_unlock(&bootMutex);
   }
}


