//-----------------------------------------------------------------------------
// File:        Canbasic.c
// Project:     CanOpen Protocol Stack
// Describtion: Access to the MCP2515 CAN chip.
// Date:        Sept 2008
// Author:      Gernot Kvas, Peter Hintenaus 
//-----------------------------------------------------------------------------
#include <string.h>
#include <linux/types.h>
#include <stdio.h>
#include <stdint.h>
#include <pthread.h>
#include <sched.h>
#include "CanBasic.h"
#include "mcp2515.h"


static MobType MobTable[MOBTABLESZ];
enum TxBuf_t {TxBuf0 = 0, TxBuf1, TxBuf2};

static errorCallbackTy errorCallback;

static MobIdTy intx = noMob, outtx = noMob, txmob = noMob;

static void SetupMob(uint32_t buf, uint32_t cobid, uint32_t msglen, char *dat) {
   uint8_t dlc = 0;
   uint8_t msg[13];

   msg[0] = (uint8_t)(cobid >> 3); msg[1] = (uint8_t)(cobid << 5);
   msg[2] = 0; msg[3] = 0;
   msg[4] = (msglen >= 0) ? (msglen & 0xF) : (1 << 6);
   msg[5] = dat[0]; msg[6] = dat[1]; msg[7] = dat[2]; msg[8] = dat[3];
   msg[9] = dat[4]; msg[10] = dat[5]; msg[11] = dat[6]; msg[12] = dat[7];
   mcp_load_tx(buf, msg);
}

/* Probably sends the message */
static void EnQTx(MobIdTy mobno) {
   // printf("EnQTX\n");
   take_lock();
   if (mobno->nxt == noMob) {
      mobno->nxt = mobno;
      if (outtx == noMob) {
         if (txmob == noMob) {
            mobno->nxt = noMob;
            txmob = mobno;
            SetupMob(TxBuf0, mobno->cobId, mobno->datalen, mobno->data.c);   //mob must be invalid            
            mcp_rts(TxBuf0);
            release_lock();
            return;
         } else {
            outtx = mobno;
         }
      } else {
         intx->nxt = mobno;
      }
      intx = mobno;
   }
   release_lock();
}

static MobIdTy DeQTx() {
   MobIdTy mobno = outtx;
   outtx = (mobno->nxt == outtx) ? noMob : mobno->nxt;
   mobno->nxt = noMob;
   return mobno;
}

static inline int EmptyTxQ() { 
   return outtx == noMob;
}

static inline int MobBusy(MobType *m) {
   return m->nxt != noMob;
}

//---------------------------------------------------------------------------------
// Hashing the MOB table
//---------------------------------------------------------------------------------
void InitMobTable() {
   int i;

   for (i = 0; i < MOBTABLESZ; i++) {
      MobTable[i].mode = unused;
   }
}

//---------------------------------------------------------------------------------
// Insert:      Insert a mob to the mob table (hash table)
// Parameter:   CobIdTy cobId.........Mob id
//              modeTy mode...........Type of the mob (hard,soft,..)
// Return:      MobIdTy mobId.........Mob
//---------------------------------------------------------------------------------
MobIdTy Insert(CobIdTy cobId, modeTy mode) {
   cobId &= 0x7FF;
   MobTable[cobId].cobId = cobId;
   MobTable[cobId].mode = mode;
   return MobTable + cobId;
}

//---------------------------------------------------------------------------------
// Find:        Find a mob in the table
// Parameter:   CobIdTy cobId.........Mob id
// Return:      MobIdTy mobId.........Mob
//--------------------------------------------------------------------------------- 
static inline MobType *Find(CobIdTy cobId) {
   cobId &= 0x7FF;
   if (MobTable[cobId].mode != deleted && MobTable[cobId].mode != unused) return MobTable + cobId;
   return NULL;
} 

//---------------------------------------------------------------------------------
//  InitMob
//---------------------------------------------------------------------------------
MobIdTy InitMob(CobIdTy cobid, modeTy mode, int usrData, void (* callback)(struct MobSt *)) {
   MobIdTy mobid;

   if (cobid & 0x80000000) return noMob;
   take_lock();
   mobid = Insert(cobid, mode);
   mobid->usrData = usrData;
   mobid->nxt = noMob;
   mobid->CallBack = callback;
   release_lock();
   return mobid;
}

//------------------------------------------------------------------------------
//  DeleteMob
//------------------------------------------------------------------------------
void DeleteMob(CobIdTy cobid) {
   MobType *mob = Find(cobid);
   if (mob == NULL) return;
   take_lock();
   mob->mode = deleted;
   release_lock();
}

//-----------------------------------------------------------------------------
// set data - for a specific message object, set the data
//-----------------------------------------------------------------------------
void SetData (MobIdTy mobid, uint8_t datalen, uint8_t * data) {
   if (mobid->mode == sftrcv || mobid->mode == deleted) return;
   mobid->datalen = datalen;
   memcpy((char *)mobid->data.c, data, datalen);
}

//-----------------------------------------------------------------------------
// Send a message
//-----------------------------------------------------------------------------
void XmitMsg(MobIdTy mobid) {
   if (mobid->mode == deleted) return;
   EnQTx(mobid);
}

//-----------------------------------------------------------------------------
// send remote frame
//-----------------------------------------------------------------------------
void XmitRmtFrm(MobIdTy mobid) {
   mobid->datalen = -1; // To tell the Tx it is a Remote Frame
   EnQTx(mobid);
}

//-----------------------------------------------------------------------------
// message interrupt
//-----------------------------------------------------------------------------
void PollCan() {
   uint8_t err;
   int intr, stat;
   static int ask0 = 1;

   uint8_t tmp[4]; // Need to be exactly 32 bit for the UIO read
   uint8_t buf[8]; // Could be shared with tmp;

/*
   int my_policy;
   struct sched_param my_param;

   pthread_getschedparam (pthread_self (), &my_policy, &my_param);

  printf ("Can ISR running at %s/%d\n",
        (my_policy == SCHED_FIFO ? "FIFO"
        : (my_policy == SCHED_RR ? "RR"
        : (my_policy == SCHED_OTHER ? "OTHER"
        : "unknown"))),
        my_param.sched_priority);
*/
   while(1) {
      read(hw->uio_fd, tmp, 4);
      
      // printf("Interrupt %d\n", tmp[0]);
      /* Read out interrupt register CAINTF */
      for(;;) {
         take_lock();
         
         if (((stat = mcp_read_stat()) & (Tx0Int_Stat | Rx0Int_Stat | Rx1Int_Stat)) == 0) {
            if ((intr = mcp_intstat()) == 0) {
               // printf("PollCan, leaving loop. stat = %x\n", stat);
               release_lock();
               break;            
            }

            if(intr & ErrInt) {
               mcp_read(MCP_EFLG, &err, 1);
               // printf("PollCan, handling local error, intr = %x, err = %x\n", intr, err);
               mcp_bitmod(MCP_CANINTF, ErrInt | WakInt | MErrInt, 0x00);
               errorCallback((err & 0xc0) ? rxovrrn : ((err & 0x20) ? busoff : ((err & 0x18) ? warn : ok)));
            }
         } else {
            int handled = 0;
            if(stat & (Rx0Int_Stat | Rx1Int_Stat)) { // We use Rollover, so both Rx may be used
               uint16_t msgid;
               uint8_t msg[13];
               uint8_t rxbctrl;
               int buffer;
               MobType  *m; 

               // printf("PollCan, handling RX data\n");
               if (ask0) buffer = (stat & Rx0Int_Stat) ? 0 : 1;
               else buffer = (stat & Rx1Int_Stat) ? 1 : 0;
               ask0 = !ask0;
               mcp_read_rx(buffer, msg); 

               msgid = msg[1] >> 5; msgid |= (msg[0] << 3);            
               rxbctrl = mcb_read_rxbctrl(buffer);
               m = Find(msgid);   // Compare the received id with the mob table
               if (m != NULL) {
                  // printf("%p %x\n", m, msgid);
                  if (rxbctrl & 0x8 && m->mode == hrdxmit) EnQTx(m);
                  else {            
                     m->datalen = msg[4];
                     m->data.c[0] = msg[5]; m->data.c[1] = msg[6]; m->data.c[2] = msg[7]; m->data.c[3] = msg[8];
                     m->data.c[4] = msg[9]; m->data.c[5] = msg[10]; m->data.c[6] = msg[11]; m->data.c[7] = msg[12];
                     m->CallBack(m);                     // Callback for this id
                  }
               }
               handled = 1;
            }
            if (stat & Tx0Int_Stat) { // Currently, we only use one Tx Buffer
               // printf("PollCan, handling TX data\n");
               mcp_bitmod(MCP_CANINTF, Tx0Int| WakInt | MErrInt, 0x00);
               txmob->CallBack(txmob);
               txmob = noMob;
               if (!EmptyTxQ()) {                     // Send all mob's in the queue
                  MobType  *m;
                  txmob = DeQTx();
                  SetupMob(TxBuf0, txmob->cobId, txmob->datalen, txmob->data.c);
                  mcp_rts(TxBuf0);
               }
               handled = 1;
            }
            if (!handled) {
               // printf("PollCan, unhandled Interrupt\n");
               // unhandled interrupt. Just get rid of it.
               mcp_bitmod(MCP_CANINTF, WakInt | MErrInt, 0x00);            
            }
         } 
         release_lock();
      }
   }
}

void InitCan(uint32_t baudrate, errorCallbackTy ecb) {
   char t0, t1;

   mcp_init("/dev/spi0", "/dev/uio0");
   mcp_reset();
   mcp_setbaud(baudrate);
   mcp_start(PollCan);

   // Do some chip setup

   errorCallback = ecb;

   take_lock();
   intx = outtx = txmob = noMob;   
   InitMobTable();
   release_lock();
}   

void ChkNRestartChip() {
   mcp_reset();
}

void ChipOff() {
   mcp_reset();
}

