#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <pthread.h>
#include <fcntl.h>
#include <sys/uio.h>
#include "chemometry.h"
#include "blobtags.h"
#include "chemoengine.h"

#ifndef SPETARGET
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#endif

//Defines data,const,global tabl
typedef struct {
    char sym[100];  //Symbol name
    int numcount;   // bei array mehr als nur 1
    int datentyp;   //datentyp laut scalar, vector....
    char type[20];  //type gibt an ob const,global oder data
    float values[5000];  //array mit den werten
    int adress;
    int busy;         //Zelle kennzeichnen ob vergeben oder nicht
} def_table;

typedef struct {
    char sym[100];  //Symbol name
    int busy;       //Zelle kennzeichnen ob vergeben oder nicht
} def_func;

#define STORE  0x40010000
#define DEFUN  0x40090000
#define CUT    0x40050003
#define RETURN 0x40080000
#define REPORT 0x400A0000


#define FUNCTIONTAG 0x300B
#define SCALARTAG 0x1001

static char *inFileName = "/dev/spe/chemoin";
static char *outFileName = "/dev/spe/chemoout";

static FILE *binfile;
static int chemoin, chemoout, EngineLoaded = 0;

static ComponentInst *inst;

static pthread_mutex_t chemoMutex;

int initChemometry(ComponentInst *i) {
   inst = i;
   return 1;
}

static char *readData(char *dest, int32_t cnt) {
   int32_t c = 0;
   while (c != cnt) {
      int r = read(chemoout, dest + c, cnt - c);
      if (r < 0) return "Chemometry: unable to read header / response / data (system)";
      c += r;
   }
   return NULL;
}

static void skipData(int32_t cnt) {
   int32_t c = 0;
   char b;
   while (c != cnt) {
      int r = read(chemoout, b, 1);
      c += r;
   }
}

static char *ChemoErrors[] = {
   "Chemometry: global table overflow",
   "Chemometry: stack overflow",
   "Chemometry: stack underflow",
   "Chemometry: type missmatch",
   "Chemometry: size missmatch",
   "Chemometry: invalid instruction",
   "Chemometry: division by zero",
   "Chemometry: index out of bounds",
   "Chemometry: call stack overflow",
   "Chemometry: call stack underflow",
   "Chemometry: function table overflow",
   "Chemometry: unknown tag",
   "Chemometry: checksum error",
   "Chemometry: timeout error"
};

static char *ParseResponse(int respFile) {
   struct {
#ifdef SPETARGET
      unsigned len;
      int ret;
      unsigned dummy1, dummy2, dummy3, dummy4;
#else
      unsigned ret;
      unsigned len;
#endif
   } header;
   char *reason = NULL;

//   fprintf(stderr, "Chemo engine: in ParseResponse\n");
   reason = readData((char *)(&header), sizeof(header));
   if (reason == NULL && header.ret < 0) {
      if (-header.ret - 1 >= 15) reason = "Chemometry: unknown error (system)"; 
      else reason = ChemoErrors[-header.ret - 1];
   }	
//   fprintf(stderr, "Chemo engine, Response =  0x%x, length = %d\n", header.ret, header.len);
   return reason;
}

void armChemometry(vm *task) {
}

int doChemometry(int32_t procNo, ComponentInst *inst, int32_t **PC, int32_t **SP, int32_t **FP, int32_t *excSP, int32_t **excHndlrPC, int32_t **excHndlrSP, int32_t **excHndlrFP) {
   ProcInterfaceDef *Interface = inst->definition->interfaces + procNo;
   int32_t parIdx, iovIdx, *actpar = *SP, cnt = 0, written = 0;
   char *reason = NULL;
   struct iovec ioV[34];
   struct {
      int32_t tag;
      int32_t len;
   } header[16];

   if (!EngineLoaded) reason = "Chemometry: no chemometric model loaded";
   else if (Interface->parcnt > 16) reason = "Chemometry: too many parameters for component call";
   else {
      for (parIdx = Interface->parcnt - 1, iovIdx = 1; parIdx >= 0; parIdx--, iovIdx += 2) {
         TypeDef *ty = Interface->params + parIdx;
         int32_t *data = actpar;
         if (ty->tag == FLOAT) {
            header[parIdx].tag = floatScalar; header[parIdx].len = 1;
            actpar++; cnt += 2;
         } else if (ty->tag == ARRAY  && ty->subs->tag == FLOAT) {
            header[parIdx].tag = floatVector; header[parIdx].len = ty->par;
            actpar += ty->par; cnt += (2 + ty->par);
         } else if (ty->tag == BLOB) {
            BlobHeader *bh = (BlobHeader *)(*actpar);
            if (bh != NULL) {
               header[parIdx].tag = bh->tag; header[parIdx].len = bh->dataCnt;
               cnt += (2 + bh->dataCnt);
               bh++; data = (int32_t *)bh;
               actpar++; 
            } else {
               reason = "Chemometry: empty blob parameter";
               break;
            }
         } else {
            reason = "Chemometry: parameter TYPE not supported";
            break;
         }
         ioV[iovIdx].iov_base = header + parIdx; ioV[iovIdx].iov_len = (header[parIdx].len == 1) ? sizeof(int32_t) : 2 * sizeof(int32_t);
         ioV[iovIdx + 1].iov_base = data; ioV[iovIdx + 1].iov_len = header[parIdx].len * sizeof(int32_t);
      }
      if (reason == NULL) {
         int32_t funcall[2] = {0x40070000 + procNo, cnt};
         ioV[0].iov_base = funcall; ioV[0].iov_len = 2 * sizeof(int32_t); cnt += 2;
         if (pthread_mutex_lock(&chemoMutex)) {
            reason = "Chemometry: cannot lock mutex (system)";
         } else {
            int xferCnt = writev(chemoin, ioV, iovIdx);
            written = 1;
            // printf("in doChemometry, xferCnt = %d, expected %d\n", xferCnt, cnt * sizeof(int32_t));
            if (xferCnt != cnt * sizeof(int32_t)) {
               reason = "Chemometry: problem sending data (system)";
            }
         }
      }
   }

   for (parIdx = Interface->parcnt - 1; parIdx >= 0; parIdx--) {
      TypeDef *ty = Interface->params + parIdx;
      releaseParameter(ty, SP);
   }
   if (reason == NULL || written) {
      struct {
#ifdef SPETARGET
         unsigned len;
         int ret;
         unsigned dummy1, dummy2, dummy3, dummy4;
#else
         unsigned ret;
         unsigned len;
#endif
      } resp;
      char *r;
            
      r = readData((char *)(&resp), sizeof(resp));
      if (reason == NULL) reason = r;
#ifdef SPETARGET
      resp.len /= sizeof(unsigned);
#endif
      // printf("in doChemometry: ret = 0x%x, len = %d\n", resp.ret, resp.len);
      if (reason == NULL && resp.ret != SUCCESS) {
         char error[100];
         if (-resp.ret - 1 >= 14) reason = "Chemometry: unknown error (system)"; 
         else reason = ChemoErrors[-resp.ret - 1];
      }
      if (reason == NULL) {
         TypeDef *ty = &(Interface->ret);
         int32_t i, cnt = 1;
         if (ty->tag == TUPPLE) {
            cnt = ty->par; ty = ty->subs;
            // printf("reading tupple\n");
         } else if (ty->tag == NOTYPE) {
            cnt = 0;
         }
         for (i = 0; i < cnt; i++, ty++) {
            if (ty->tag == FLOAT) {
               struct {int32_t tag, dat;} r;
               // printf("reading float\n");
               if (resp.len < 2) {
                  reason = "Chemometry: tupple length missmatch in return";
                  if (resp.len > 0) skipData(resp.len * sizeof(unsigned));
                  break;   
               }
               reason = readData((char *)(&r), sizeof(r));
               if (reason == NULL) {
                  resp.len -= 2;
                  if (r.tag == floatScalar) *(--(*SP)) = r.dat;
                  else reason = "Chemometry: type missmatch in return";
               }               
            } else if (ty->tag == BLOB) {
               struct {int32_t tag, len;} r;
               if (resp.len < 2) {
                  reason = "Chemometry: tupple length missmatch in return";
                  if (resp.len > 0) skipData(resp.len * sizeof(unsigned));
                  break;   
               }
               reason = readData((char *)(&r), sizeof(r));
               if (reason == NULL) {
                  resp.len -= 2;
                  BlobHeader *blHd = allocBlob(r.len);
                  if (resp.len < r.len) {
                     reason = "Chemometry: tupple length missmatch in return";
                     if (resp.len > 0) skipData(resp.len * sizeof(int32_t));
                     freeBlob(blHd);
                     break;   
                  }
                  reason = readData((char *)(blHd->data), r.len * sizeof(int32_t));
                  if (reason != NULL) freeBlob(blHd);
                  else {
                     blHd->tag = r.tag;
                     // printf("allocated %x\n", (int)data);
                     *(--(*SP)) = (int32_t)(blHd);
                     resp.len -= r.len;
                  }
               }
            } else if (ty->tag == ARRAY && ty->subs->tag == FLOAT) {
               struct {int32_t tag, len;} r;
               char *lReason;
               if (resp.len < 2) {
                  reason = "Chemometry: return value expected";
                  if (resp.len > 0) skipData(resp.len * sizeof(unsigned));
                  break;   
               }
               reason = readData((char *)(&r), sizeof(r));
               if (reason == NULL) {
                  // printf("in doChemometry, ty->par = %d\n", ty->par);
                  if (ty->par != r.len) {
                     static char e[100];
                     sprintf(e, "Chemometry: size missmatch in returned ARRAY, expected %d, actual %d", ty->par, r.len);
                     reason = e;
                  }
                  resp.len -= 2;
                  (*SP) -= r.len;
                  // printf("in doChemometry, array size is %d, SP = %x\n", r.len, *SP);
                  if (resp.len < r.len) {
                     reason = "Chemometry: length missmatch in return";
                     if (resp.len > 0) skipData(resp.len * sizeof(unsigned));
                     break;   
                  }
                  lReason = readData((char *)(*SP), r.len * sizeof(int32_t));
                  if (reason == NULL) reason = lReason;
                  resp.len -= r.len; 
               } 
            } else {
               reason = "Chemometry: return TYPE not supported";
            }
            if (reason != NULL) {
               int32_t j;
               for (j = i-1; j >= 0; j--) releaseParameter(--ty, SP);
               break;
            }
         }
      }
      if (reason == NULL && resp.len != 0) reason = "Chemometry: length missmatch in return";
      pthread_mutex_unlock(&chemoMutex);
   }

   if (reason != NULL) {
      return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0xfe, reason);
   }
   return 1;   
}

int initChemometryEngine(ComponentInst *inst) {
   int rc;
   if (strcmp(inst->instName, "FtIrChemometryEngine")) return 0;
   EngineLoaded = 0;
   rc = pthread_mutex_init(&chemoMutex, NULL);
   if (rc) {
      fprintf(stderr, "initChemometryEngine: chemoMutex");
      return 0;
   }   
   return 1;
}

void armChemometryEngine(vm *task) {
}

static char *loadEngine (char *model, int version) {
   int funtabsiz, vartabsiz;
   int flen, j, rc;
   unsigned *fun;
   long funpos;
   int instr, arg[2], eof, defun;
   def_table sym;
   char binName[128];
   char versStr[20];
   char msg[100];

   strcpy(binName, projDir);
   strcat(binName, model);
   if (version < 0 || version > 9999) return "ChemometryEngine: model version out of range";
   sprintf(versStr, ".%04d", version);
   strcat(binName, versStr);
   sprintf(msg, "loading model: %s", binName);
   logError(0xFF, msg);

   if ((binfile = fopen(binName, "rb")) == NULL) return "ChemometryEngine: cannot open model";
   pthread_mutex_lock(&chemoMutex); 
   if (EngineLoaded) {
   } else {
#ifdef SPETARGET
      // fprintf(stderr, "Chemo engine: opening chemoin & chemoout\n");
      if ((chemoin = open(inFileName, O_WRONLY)) < 0) {
         pthread_mutex_unlock(&chemoMutex);
         return "ChemometryEngine, cannot open engine";
      }
      if ((chemoout = open(outFileName, O_RDONLY)) < 0) {
         close(chemoin);
         pthread_mutex_unlock(&chemoMutex);
         return "ChemometryEngine, cannot open engine";
      }
      // fprintf(stderr, "Chemo engine: done\n");
#else
      struct sockaddr_in peer;
      peer.sin_family = AF_INET;
      peer.sin_addr.s_addr = inet_addr("127.0.0.1");
      peer.sin_port = htons(4039);
      chemoin = chemoout = socket(PF_INET, SOCK_STREAM, 0);
      connect(chemoout, &peer, sizeof(struct sockaddr_in));
#endif
   }
		
   fread(&vartabsiz, sizeof(int), 1, binfile);

   for (j = 0; j < vartabsiz; j++) {
      fread(&sym, sizeof(def_table), 1, binfile);
      if (strcmp(sym.type, "data") == 0) {
         int dlen = (sym.datentyp == SCALARTAG) ? 2 : (2 + sym.numcount);
         char *resp;
         fun = malloc((2 + dlen) * sizeof(unsigned));
         fun[0] = STORE + j; fun[1] = dlen;
         fun[2] = sym.datentyp; 
         if (sym.datentyp == SCALARTAG) 
            fun[3] = *((unsigned *)&(sym.values[0]));
         else {
            int i;
            fun[3] = sym.numcount;
            for (i = 0; i < sym.numcount; i++) fun[4 + i] = *((unsigned *)&(sym.values[i]));
         }
         fprintf(stderr, "Chemo engine: loading data...length = %d\n", dlen - ((sym.datentyp == SCALARTAG) ? 1 : 2));
         if (write(chemoin, fun, sizeof(int) * (dlen + 2)) != sizeof(int) * (dlen + 2)) {
            fclose(binfile);
            if (chemoout != chemoin) close(chemoout);
            close(chemoin);
            EngineLoaded = 0;
            pthread_mutex_unlock(&chemoMutex);
            return "ChemometryEngine: error writing";
         }
         if ((resp = ParseResponse(chemoout)) != NULL) {
            fclose(binfile);
            if (chemoout != chemoin) close(chemoout);
            close(chemoin);
            EngineLoaded = 0;
            pthread_mutex_unlock(&chemoMutex);
            return resp;
         }
         free(fun); 	
      }
   }
   fread(&funtabsiz, sizeof(int), 1, binfile);

   fseek(binfile, sizeof(def_func) * funtabsiz, SEEK_CUR);

   eof = 0;
   for(;;) {
      char *resp;
      for (;;) {
         if (feof(binfile)) {
            eof = 1;
            break;
         }
         fread(&instr, sizeof(int), 1, binfile);
         if ((instr & 0xFFFF0000) == DEFUN) break;
      }
      if (eof)  break;
      defun = instr;
      funpos = ftell(binfile); flen = 0;
      for (;;) {
         if (feof(binfile)) {
            eof = 1;
            break;
         }
         fread(&instr, sizeof(int), 1, binfile);
         // fprintf(stderr, "   0x%8x\n", instr);
         flen++;
         if (instr == CUT) {
            flen += 2;
            fread(arg, sizeof(int), 2, binfile);
            // fprintf(stderr, "   %8d\n    %8d\n", arg[0], arg[1]);
         } else if ((instr & 0xFFFF0000) == REPORT || instr == RETURN) break; 
      }
      if (eof) break;		
      fun = malloc((2 + 2 + flen) * sizeof(int));
      if (fun == NULL) {
         if (chemoout != chemoin) close(chemoout);
         fclose(binfile);
         close(chemoin);
         EngineLoaded = 0;
         pthread_mutex_unlock(&chemoMutex);
         return "ChemometryEngine: out of memory";
      }
      fprintf(stderr, "Chemo engine: function definition %8x, length %d\n", defun, flen);
      fun[0] = defun; fun[1] = flen + 2; 
      fun[2] = FUNCTIONTAG; fun[3] = flen;
      fseek(binfile, funpos, SEEK_SET);
      fread(&(fun[4]), sizeof(int), flen, binfile);
      if (write(chemoin, fun, sizeof(int) * (flen + 4)) != sizeof(int) * (flen + 4)) {
         if (chemoout != chemoin) close(chemoout);
         fclose(binfile);
         close(chemoin);
         EngineLoaded = 0;			
         pthread_mutex_unlock(&chemoMutex);
         return "ChemometryEngine: error writing";
      }
      if ((resp = ParseResponse(chemoout)) != NULL) {
         fclose(binfile);
         if (chemoout != chemoin) close(chemoout);
         close(chemoin);
         EngineLoaded = 0;
         pthread_mutex_unlock(&chemoMutex);
         return resp;
      }
      free(fun);	
   }
   fclose(binfile);
   EngineLoaded = 1;
   pthread_mutex_unlock(&chemoMutex);
   return NULL;
}

int doChemometryEngine(int32_t procNo, ComponentInst *inst, int32_t **PC, int32_t **SP, int32_t **FP, int32_t *excSP, int32_t **excHndlrPC, int32_t **excHndlrSP, int32_t **excHndlrFP) {
   switch (procNo) {
      case 0: {
         int version;
         char *model, *reason;
         version = **SP; (*SP)++;
         model = (char *)(Memory[**SP]); (*SP)++;
         reason = loadEngine(model, version);
         if (reason != NULL) {
            return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0xFF, reason);
         }
         break;
      }
   }
   return 1;
}
