#include <stdio.h>
#include <fcntl.h>
#include <stdint.h>
#include <string.h>
#include <stdlib.h>
#include <pthread.h>
#include <sys/types.h>
#include <dirent.h>
#include <unistd.h>
#include "datastore.h"
#include "dynamicdefs.h"

#define DICTSIZE (256 * 5)

/*
Storage Format:

	Storage = {Metadata}{Dataset}.
	MetaData = "SetId32" {"String" "Type32"}.
	Dataset = Header {Entry}.
	Header = "SetId32" "Length32" "time" "timestamp32".
	Entry = ("Data32" | Blob).
	Blob = "BlobId32" "DataLen32" {"Data32"}.
*/

struct StoreSpec {
   ComponentInst *inst;
   char *storeName;
   int32_t oNumber, mn_o, mx_o;
   int32_t mx_len, len, mx_blks;
   int out;
   pthread_mutex_t mutex;
   int32_t setNo;
   struct dataSet *set;
}; 

int initDataStore(ComponentInst *inst) {
   struct StoreSpec *s = malloc(sizeof(struct StoreSpec));
   int rc, l;
   DIR *storeDir;
   if (s == NULL) return 0;
   s->inst = inst;
   s->setNo = 0;
   s->set = NULL;
   inst->instSpecific = s;
   l = strlen(storeRoot);
   s->storeName = malloc(l + strlen(inst->instName) + 10);
   if (s->storeName == NULL) return 0;
   strcpy(s->storeName, storeRoot);
   if (l > 0 && storeRoot[l-1] != '/') {
      s->storeName[l] = '/'; s->storeName[l + 1] = '\0';
   }
   strcat(s->storeName, inst->instName);
   s->mn_o = INT32_MAX; s->mx_o = -1;
   storeDir = opendir(s->storeName);
   if (storeDir != NULL) {
      struct dirent *de;
      for (;;) {
         int32_t no;
         de = readdir(storeDir);
         if (de == NULL) break;
         if (strcmp(".", de->d_name) == 0) continue;
         if (strcmp("..", de->d_name) == 0) continue;
         if (sscanf(de->d_name, "%d", &no) != 1) return 0;
         if (no > s->mx_o) s->mx_o = no;
         if (no < s->mn_o) s->mn_o = no;
      }
      if (s->mn_o == INT32_MAX) {
         s->oNumber = -1; s->mn_o = s->mx_o = 0;
      } else {
         s->oNumber = s->mx_o;
      }
   } else {   
      mkdir(s->storeName, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH);
      s->oNumber = -1; s->mn_o = s->mx_o = 0;
   }
   strcat(s->storeName, "/%d");
   rc = pthread_mutex_init(&(s->mutex), NULL); // Mutex for condition
   if (rc) {
      fprintf(stderr, "initDataStore, pthread_mutex_init");
      exit(EXIT_FAILURE);
   }   
   s->out = -1;
   s->len = 0; s->mx_len = 10000000; s->mx_blks = 100;
   return 1;
}

void armDataStore(vm *task) {}

static char *writeMeta(char *data, int len, void *spec) {
   struct StoreSpec *s = (struct StoreSpec *)spec;
   int i;
   char *r;
   if (s->out == -1) return "DataStore: inconsistent state";
   while (len > 0) {
      int l = write(s->out, data, len);
      if (l < 0) return "DataStore: cannot write metadata";
      len -= l;
   }
   return NULL;
}

static void dumpDefs(struct StoreSpec *s) {
   int i;
   for (i = 0; i < s->setNo; i++) {
      if (s->set[i].setSize > 0) {
         int j;
         printf("id: %d\n", s->set[i].header.id);
         for (j = 0; j < s->set[i].setSize; j++) {
            printf("name: %s, type: %x, data: %x\n", s->set[i].entry[j].meta.name, s->set[i].entry[j].meta.type, s->set[i].entry[j].data);
         }
      }
   }
}

static char *writeFile(struct iovec *iov, int32_t cnt, void *spec) {
   struct StoreSpec *s = (struct StoreSpec *)spec;
   char *r = NULL;
   int i, l, c;
   int32_t setCnt;
   pthread_mutex_lock(&s->mutex);
   if (s->out == -1) {
      char fName[512];
      s->oNumber++; s->mx_o = s->oNumber;
      if (s->mx_o - s->mn_o >= s->mx_blks) {		// make store circular
         sprintf(fName, s->storeName, s->mn_o);
         unlink(fName); 
         s->mn_o++;
      }
      sprintf(fName, s->storeName, s->oNumber); 
      fprintf(stderr, "Datastore, opening %s\n", fName);
      s->out = open(fName, O_WRONLY | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
      for (i = 0, setCnt = 0; i < s->setNo; i++)
         if (s->set[i].setSize > 0) setCnt++;
      c = write(s->out, &setCnt, sizeof(int32_t));
      if (c < 0) {
         close(s->out);
         unlink(fName);
         pthread_mutex_unlock(&s->mutex);
         return "DataStore: cannot write metadata";
      }
      for (i = 0; i < s->setNo; i++) {
         if (s->set[i].setSize > 0) {
            r = doOutputMetaData(s->set + i, s, writeMeta);
            if (r != NULL) {
               close(s->out);
               unlink(fName);
               pthread_mutex_unlock(&s->mutex);
               return r;
            }
         }
      }
      s->len = 0;
   }
   for (i = 0; i < cnt; i++) {
      l = 0;
      while (l < iov[i].iov_len) {
         if ((c = write(s->out, iov[i].iov_base + l, iov[i].iov_len - l)) < 0) r = "DataStore: cannot write";
         l += c; s->len += c;
      }
   }
   /*
   if ((c = writev(s->out, iov, cnt)) < 0) r = "DataStore: cannot write";
   s->len += c;
   */
   if (s->len > s->mx_len) {
      close(s->out);
      s->out = -1;
   }
   pthread_mutex_unlock(&s->mutex);      
   return r;
}

static int readFile(char *buffer, int len, void *par, char **ex) {
   int f = *((int*)par);
   char *b = NULL;
   if (buffer == NULL) buffer = b = malloc(len);
   while (len > 0) {
      int l = read(f, buffer, len);
      if (l <= 0 && b != NULL) free(b);
      if (l == 0) return 1;
      if (l < 0) {
         *ex = "DataStore: cannot read";
         return 0;
      }
      buffer += l; len -= l;
   }
   return 0;
}

static char *LatestInput(struct dataSet *set, struct StoreSpec *spec) {
   int dataRead = 0;
   char *ex = NULL;
   int i;

   for (i = spec->mx_o; i >= spec->mn_o; i--) {
      char fName[512], *b;
      int f, eof, metaRead = 0, j;
      int32_t metaCnt;
      sprintf(fName, spec->storeName, i);
      f = open(fName, O_RDONLY);
      if (f == -1) continue;
      eof = readFile((char *)(&metaCnt), sizeof(metaCnt), &f, &ex);
      if (eof) {
         close(f);
         continue;
      }
      if (ex != NULL) {
         close(f);
         continue;
      }
      for (j = 0; j < metaCnt; j++) {
         int metaOk = MetaDataValid(set, &f, readFile);
         if (metaOk == -1) {
            metaRead = 0; break;
         }
         if (metaOk == 1) metaRead = 1;
      }
      if (!metaRead) {
         close(f);
         continue;
      }
      // read the data
      for (;;) {
         int rd = doInput(set, &f, readFile, &ex);
         if (rd == -1) break;
         if (ex != NULL) return ex;
         if (rd == 1) dataRead = 1;
      }
      close(f);
      if (dataRead) break;
   }
   if (ex != NULL && !dataRead) return "DataStore: readLatestDataset, no dataset found";
   return ex;
}

int doDataStore(int32_t procNo, ComponentInst *inst, int32_t **PC, int32_t **SP, int32_t **FP, int32_t *excSP, int32_t **excHndlrPC, int32_t **excHndlrSP, int32_t **excHndlrFP) {
   switch (procNo) {
      case 0:    // PROCEDURE addBoolToDataset(VAR BOOL data, INTEGER SetId);
      case 1:    // PROCEDURE addIntegerToDataset(VAR INTEGER data, INTEGER SetId);
      case 2:    // PROCEDURE addTimestampToDataset(VAR TIMESTAMP data, INTEGER SetId);
      case 3:    // PROCEDURE addFloatToDataset(VAR FLOAT data, INTEGER SetId);
      case 4:    // PROCEDURE addBlobToDataset(VAR BLOB data, INTEGER SetId);
      {
         int32_t SetId, *dataAddr;
         char *errtxt;
         struct StoreSpec *spec = (struct StoreSpec *)inst->instSpecific;
         SetId = **SP; (*SP)++;
         dataAddr = Memory + **SP; (*SP)++;
         if (SetId < 0) return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x20, "DataStore: invalid SetId");
         if (dataAddr < spec->inst->objDict || spec->inst->objDict + DICTSIZE <= dataAddr) 
            return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x20, "DataStore: variable not in OBJECTDICTIONARY");
         if (spec->setNo - 1 < SetId) {
            struct dataSet *s = realloc(spec->set, (SetId + 1) * sizeof(struct dataSet));
            int32_t i;
            if (s == NULL) return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x20, "DataStore, cannot allocate dataSet definitions");
            for (i = spec->setNo; i <= SetId; i++) {
               s[i].setSize = 0; s[i].header.id = i;
            }
            spec->setNo = SetId + 1;
            spec->set = s;
         }
         if ((errtxt = addEntry2Set(dataAddr, &(spec->set[SetId]))) != NULL) {
            return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x20, errtxt);
         }			
         break;
      }
      case 5: {   // PROCEDURE writeDataset(INTEGER SetId);
         int32_t SetId;
         char *errtxt;
         struct StoreSpec *spec = (struct StoreSpec *)inst->instSpecific;
         // dumpDefs(spec);
         SetId = **SP; (*SP)++;
         errtxt = doOutput(spec->set + SetId, spec, writeFile);
         if (errtxt != NULL) return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x20, errtxt);
         // dumpDefs(spec);
         break;
      }
      case 6: {   // PROCEDURE readLatestDataset(INTEGER SetId);
         int32_t SetId;
         char *errtxt;
         struct StoreSpec *spec = (struct StoreSpec *)inst->instSpecific;
         // dumpDefs(spec);
         SetId = **SP; (*SP)++;
         errtxt = LatestInput(spec->set + SetId, spec); 
         if (errtxt != NULL) return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x20, errtxt);
         // printf("readLatest\n");
         // dumpDefs(spec);
         break;
      }
      case 7: {   // PROCEDURE configure(INTEGER BlkSize, INTEGER BlkCount);
         int32_t BlkSize, BlkCount;
         struct StoreSpec *spec = (struct StoreSpec *)inst->instSpecific;
         BlkCount = **SP; (*SP)++;
         BlkSize = **SP; (*SP)++;
         if (BlkSize > 0) spec->mx_len = BlkSize;
         if (BlkCount > 0) spec->mx_blks = BlkCount;
         break;    
      }
      case 8: {   // PROCEDURE checkpoint()
         struct StoreSpec *s = (struct StoreSpec *)inst->instSpecific;
         pthread_mutex_lock(&s->mutex);
         close(s->out);
         s->out = -1;
         pthread_mutex_unlock(&s->mutex);      
      }
   }
   return 1;
}


