#include <stdio.h>
#include <unistd.h>
#include <sys/poll.h>
#include <sys/ioctl.h>
#include <stdint.h>
#include <string.h>
#include <fcntl.h>
#include <pthread.h>
#include <sched.h>
#include <mcheck.h>
#include "spectrometer.h"
#include "blobtags.h"
#include "../../speconfig/speconfiglib.h"

#ifndef SPETARGET
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#endif

#define NOEVENTS 11
#define MIRRORCEVENTNO 8
#define STATUSEVENTNO 9
#define WATCHDOGEVENTNO 10

#define DSPCOMM_IOC_MAGIC 'k'

#define DSPCOMM_IOCRESET _IO(DSPCOMM_IOC_MAGIC, 0)
#define DSPCOMM_IOCGTIMESTAMP _IOR(DSPCOMM_IOC_MAGIC, sizeof(struct timespec), char)

static char *speEvFiles[NOEVENTS - 1] = {
   "/dev/spe/spec", "/dev/spe/raw", "/dev/spe/inter",
   "/dev/spe/lasert", "/dev/spe/laserp", "/dev/spe/laserc", "/dev/spe/dett", "dev'spe/detc", "/dev/spe/mirrorc", 
   "/dev/spe/watchdog"
};

static int evFileDesc[NOEVENTS] = {0};

static char *speCtrl = "/dev/spe/ctrl";
static int ctrl = -2;

static int32_t dsp2stmp(int32_t dsp);

int initSpectrometer(ComponentInst *inst) {
   if (strcmp(inst->instName, "FtIr")) return 0;
   inst->instSpecific = NULL;
#ifdef SPETARGET
   if (ctrl == -2) ctrl = open(speCtrl, O_WRONLY);
#endif
   return 1;
}

static void spectroTrigger(vm *task) {
   int32_t eventNo = task->triggerIdx;
   int32_t header[6];   // length in bytes, two timestamps, two wave numbers 
                        // or mirror offset and mirror amplitude and direction
   int32_t l, c;
   char *p;
   BlobHeader *blHd;
	
   do {
      l = sizeof(header); p = (char*)header;
      while (l > 0) {
         c = read(evFileDesc[eventNo], p, l); p += c; l -= c;
      }
   } while (header[0] == 0); 
   blHd = (BlobHeader *)(task->statics[4]);
   l = header[0];
   if (blHd == NULL || blHd->refCnt != 1 || blHd->dataCnt != l / sizeof(int32_t)) {
      WRITELOCK;
      dwnRefCnt(task->statics + 4);
      UNLOCK;
      // printf("spectroTrigger, allocating Blob, size: %d\n", l / 4);
      blHd = allocBlob(l / sizeof(int32_t));
      task->statics[4] = (int32_t)blHd;
   }
   p = (char *)(blHd->data);
   while (l > 0) {
      c = read(evFileDesc[eventNo], p, l); p += c; l -= c;
   }
   blHd->tag = (eventNo < 2) ? spectrum : interferogram;

   task->statics[0] = dsp2stmp(header[3]);
   task->statics[1] = dsp2stmp(header[4]);
   task->statics[2] = header[1];
   task->statics[3] = header[2];
   task->statics[5] = header[5];
}

static void interTrigger(vm *task) {
   int32_t eventNo = task->triggerIdx;
   int32_t header[6];   // length in bytes, two dummies, two timestamps, one dummy
   int32_t l, c;
   char *p;
   BlobHeader *blHd;
	
   do {
      l = sizeof(header); p = (char*)header;
      while (l > 0) {
         c = read(evFileDesc[eventNo], p, l); p += c; l -= c;
      }
   } while (header[0] == 0); 

   l = header[0];
   // printf("interTrigger, allocating Blob, size: %d\n", l / 4);
   blHd = allocBlob(l / sizeof(int32_t));
   p = (char *)(blHd->data);
   while (l > 0) {
      c = read(evFileDesc[eventNo], p, l); p += c; l -= c;
   }
   blHd->tag = interferogram;
   // if (mprobe(blHd) != MCHECK_OK) {
   //    printf("Memory corrupted in spectroTrigger\n");
   // }
   task->statics[0] = dsp2stmp(header[3]);
   task->statics[1] = dsp2stmp(header[4]);
   WRITELOCK;
   dwnRefCnt((task->statics + 2));
   UNLOCK;
   task->statics[2] = (int32_t)blHd;
}

static void monitorTrigger(vm *task) {
   int32_t eventNo = task->triggerIdx;
   int32_t header[6];   // length in bytes, five dummies
   int32_t l, c;
   char *p, *d;
	
   do {
      l = sizeof(header); p = (char*)header;
      while (l > 0) {
         c = read(evFileDesc[eventNo], p, l); p += c; l -= c;
      }
   } while (header[0] == 0); 
   if (eventNo == MIRRORCEVENTNO) l = 127 * 8 * sizeof(int32_t);
   else l = 127 * sizeof(int32_t);
   p = (char *)(task->statics);
   while (l > 0) {
      c = read(evFileDesc[eventNo], p, l); p += c; l -= c;
   }
}

static void statusTrigger(vm *task) {
   int32_t eventNo = task->triggerIdx;
   int32_t header[6];   // length in bytes, system status, dummy, dsptime, tv_sec, tv_nsec
   int32_t l, c;
   char *p;
	
   l = 6 * sizeof(int32_t); p = (char*)header;
   while (l > 0) {
      c = read(evFileDesc[eventNo], p, l); p += c; l -= c;
   }
   task->statics[0] = now();        // timestamp
   task->statics[1] = header[1];    // system status
}

static struct pollfd pollwd;
#define WDTIMEOUT 60000		// watchdog timeout in mS

#define SYSTOGGLEBIT  0x00000001
#define SYSWASDARK    0x00000002
#define SYSHASCLIPPD  0x00000004
#define SYSISSTOPPD   0x00000008
#define SYSP1SEEN     0x00000010

static void watchdogTrigger(vm *task) {
   int32_t eventNo = task->triggerIdx;
   int32_t wdok, interNotSeen = 0, reason;

   do {
      wdok = 1;
      poll(&pollwd, 1, WDTIMEOUT);
      if (pollwd.revents & POLLIN) {
         int32_t header[6];
         int32_t l, c, status;
         char *p;
         l = 6 * sizeof(int32_t); p = (char*)header;
         while (l > 0) {
            c = read(evFileDesc[eventNo], p, l); p += c; l -= c;
         }
         status = header[1];
         if ((!(status & SYSISSTOPPD)) && (!(status & SYSP1SEEN))) {
            interNotSeen++;
            if (interNotSeen >= 400) {
               wdok = 0; reason = 0;
            }
         } else interNotSeen = 0;
      } else {
         wdok = 0; reason = 1;
      }
   } while (wdok);
   task->statics[0] = reason;
}

void armSpectrometer(vm *task) {
   int32_t eventNo = task->triggerIdx;
   ComponentInst *inst = task->triggerInst;
   if (inst->events[eventNo] != NULL) {
#ifdef SPETARGET
      if (eventNo == WATCHDOGEVENTNO) 
		pollwd.fd = evFileDesc[eventNo] = open(speEvFiles[STATUSEVENTNO], O_RDONLY);
      else 
		evFileDesc[eventNo] = open(speEvFiles[eventNo], O_RDONLY);
#else
      struct sockaddr_in peer;
      peer.sin_family = AF_INET;
      peer.sin_addr.s_addr = inet_addr("127.0.0.1");
      peer.sin_port = htons(4038);
      evFileDesc[eventNo] = socket(PF_INET, SOCK_STREAM, 0);
      connect(evFileDesc[eventNo], &peer, sizeof(struct sockaddr_in));
#endif
      if (eventNo < 2) task->triggerFun = spectroTrigger;
      else if (eventNo == 2) task->triggerFun = interTrigger;
      else if (eventNo < STATUSEVENTNO) task->triggerFun = monitorTrigger;
      else if (eventNo == STATUSEVENTNO) task->triggerFun = statusTrigger;
      else if (eventNo == WATCHDOGEVENTNO) {
         pollwd.events = POLLIN;
         task->triggerFun = watchdogTrigger;
      }
   }
}

static int writeCmd(int32_t cmd[2]) {
   char *p;
   int l, c;

   p = (char *) cmd; l = 2 * sizeof(int32_t);
   while (l > 0) {
      if ((c = write(ctrl, p, l)) < 0) return -1;
      p += c; l -= c;
   }
   return 0;
}

#define setIrGain 0
#define configInterferogramAverager 1
#define setAverageCount 2
#define setRawSkips 3
#define setMeasurementSelection 4
#define setFilterWindow 5
#define StopMeasure 6
#define StartMeasure 7
#define Reset 8
#define SwitchMirrorControl 9

int doSpectrometer(int32_t procNo, ComponentInst *inst, int32_t **PC, int32_t **SP, int32_t **FP, int32_t *excSP, int32_t **excHndlrPC, int32_t **excHndlrSP, int32_t **excHndlrFP) {
   int32_t cmd[2];
   switch (procNo) {
      case setIrGain: {
         int32_t gain = **SP;
         (*SP)++;
         if (gain > 15) gain = 15;
         if (gain < 0) gain = 0;
         cmd[0] = 0x90;
         cmd[1] = gain;
         writeCmd(cmd);
         break;
      }
      case configInterferogramAverager: {
         int32_t darkLimit, clipLimit, directionSel, maxMiss, avrgCnt, corrMaxF, corrMinF, corrLen;
         darkLimit = **SP; (*SP)++; clipLimit = **SP; (*SP)++;  directionSel = **SP; (*SP)++;
         maxMiss = **SP; (*SP)++; avrgCnt = **SP; (*SP)++; corrMaxF = **SP; (*SP)++;  corrMinF = **SP; (*SP)++; corrLen = **SP; (*SP)++;
         cmd[0] = 0x9C; cmd[1] = darkLimit; writeCmd(cmd);
         cmd[0] = 0x9B; cmd[1] = clipLimit; writeCmd(cmd);
         cmd[0] = 0x9A; cmd[1] = directionSel; writeCmd(cmd);
         cmd[0] = 0x95; cmd[1] = maxMiss; writeCmd(cmd);
         cmd[0] = 0x94; cmd[1] = avrgCnt; writeCmd(cmd);
         cmd[0] = 0x93; cmd[1] = corrMaxF; writeCmd(cmd);
         cmd[0] = 0x92; cmd[1] = corrMinF; writeCmd(cmd);
         cmd[0] = 0x91; cmd[1] = corrLen; writeCmd(cmd);
         break;
      }
      case setAverageCount: {
         int32_t c = **SP; (*SP)++;
         cmd[0] = 0x94; cmd[1] = c; writeCmd(cmd);
         break;
      }
      case setRawSkips: {
         int32_t s = **SP; (*SP)++;
         cmd[0] = 0x96; cmd[1] = s; writeCmd(cmd);
         break;         
      }
      case setMeasurementSelection: {
         int32_t ms = **SP; (*SP)++;
         cmd[0] = 0x97; cmd[1] = ms; writeCmd(cmd);
         break;         
      }   
      case setFilterWindow: {
         int32_t hiWN, loWN;
         hiWN = **SP; (*SP)++; loWN = **SP; (*SP)++;
         cmd[0] = 0x99; cmd[1] = hiWN; writeCmd(cmd);
         cmd[0] = 0x98; cmd[1] = loWN; writeCmd(cmd);         
         break;
      }
      case StopMeasure:
         cmd[0] = 0x9E; cmd[1] = 1; writeCmd(cmd);
         break;
      case StartMeasure: 
         cmd[0] = 0x9E; cmd[1] = 0; writeCmd(cmd);
         break;
      case Reset: {
         FILE *dsp, *cmd;
         dsp = fopen("/dev/spe/ctrl", "wb");
         cmd = fopen("/etc/spectro/current.spectro", "r");
         if(dsp == NULL || cmd == NULL) {
            logError(0xFF, "Error resetting DSP's");
            break;
         }
         logError(0xFF, "resetting DSP's");
         if (ioctl(ctrl, DSPCOMM_IOCRESET) < 0) {
            logError(0xFF, "Error resetting DSP's");
            break;
	 }
         InitDsp(dsp, cmd);
         fclose(dsp); fclose(cmd);
         break;
      }
      case SwitchMirrorControl: {
         int32_t on = **SP; (*SP)++;
         cmd[0] = 0xA0; cmd[1] = on; writeCmd(cmd);
         break;
      }
   }
   return 1;
}

static pthread_t stamperThread;
static pthread_mutex_t stamperMutex;
static int statfd;
static int32_t lst_dsp, lst_sec, lst_ns;
static int nsPdsp = 125000;
static int32_t lst_ts = 0x80000000;

static void *TimeStamper(void *arg) {

   for (;;) {
      int32_t buffer[6], i;
      int32_t dsp, sec, ns, d_t, d_dsp, nsPd;
      char *b;
      i = 6 * sizeof(int32_t);
      b = (char *)buffer;
      while (i > 0) {
         int32_t l = read(statfd, b, i); b += l; i -= l;          
      }
      dsp = buffer[3]; sec = buffer[4]; ns = buffer[5];
      d_t = (sec - lst_sec) * 1000000000 + ns - lst_ns;
      d_dsp = dsp - lst_dsp; 
      nsPd = (d_t + (d_dsp >> 1)) / d_dsp;
      pthread_mutex_lock(&stamperMutex);
      lst_ts += (sec - lst_sec) * 1000 + (ns - lst_ns) / 1000000;
      lst_dsp = dsp; lst_sec = sec; lst_ns = ns;  
      nsPdsp += (nsPd >> 6) - (nsPdsp >> 6);
      pthread_mutex_unlock(&stamperMutex);
   }
}

static int32_t dsp2stmp(int32_t dsp) {
   int32_t ld, lts, nsPd;

   pthread_mutex_lock(&stamperMutex);
   ld = lst_dsp; lts = lst_ts; nsPd = nsPdsp;
   pthread_mutex_unlock(&stamperMutex);
   return lts + (dsp - ld) * nsPd / 1000000;
}

void internalInitFtIrClock() {
   static int stamperCreated = 0;
   if (ctrl == -2) {
      ctrl = open(speCtrl, O_WRONLY);
   }
   if (!stamperCreated) {
      int32_t buffer[6], i;
      char *b;
      statfd = open(speEvFiles[STATUSEVENTNO], O_RDONLY);
      if (statfd < 0) {
         perror("internalInitFtIrClock, spectrometer status file");
         exit(EXIT_FAILURE);
      }
      i = 6 * sizeof(int32_t);
      b = (char *)buffer;
      while (i > 0) {
         int32_t l = read(statfd, b, i); b += l; i -= l;          
      }
      lst_dsp = buffer[3]; lst_sec = buffer[4]; lst_ns = buffer[5];      
      if (pthread_mutex_init(&stamperMutex, NULL)) {
         fprintf(stderr, "internalInitFtIrClock, stamperMutex\n");
         exit(EXIT_FAILURE);
      }
      if (pthread_create(&stamperThread, &vmAttrs, TimeStamper, NULL)) {
         fprintf(stderr, "internalInitFtIrClock, stamperThread\n");
         exit(EXIT_FAILURE);
      }
      if (pthread_detach(stamperThread)) { 
         fprintf(stderr, "internalInitFtIrClock, detatching stamperThread\n");
         exit(EXIT_FAILURE);
      }
      stamperCreated = 1;
   }
}

int initTimeStampClock(ComponentInst *inst) {
   if (strcmp(inst->instName, "FtIrClock")) return 0;
   inst->instSpecific = NULL;
#ifdef SPETARGET
   internalInitFtIrClock();
#endif
   return 1;
}

int32_t now() {
   struct timespec r;
   int t;
   ioctl(ctrl, DSPCOMM_IOCGTIMESTAMP, &r);
   pthread_mutex_lock(&stamperMutex);
   t = lst_ts + (r.tv_sec - lst_sec) * 1000 + (r.tv_nsec - lst_ns) / 1000000;
   pthread_mutex_unlock(&stamperMutex); 
   return t;
}

static vm *wrapAroundTask = NULL;

void armTimeStampClock(vm *task) {
   int32_t eventNo = task->triggerIdx;
   ComponentInst *inst = task->triggerInst;
   if (inst->events[eventNo] != NULL) {
      task->triggerFun = stdTriggerFun;
      wrapAroundTask = task;
   }
}

// about one day before timestamp wraparound...

#define ALARM_THRESHOLD 2061083648

int doTimeStampClock(int32_t procNo, ComponentInst *inst, int32_t **PC, int32_t **SP, int32_t **FP, int32_t *excSP, int32_t **excHndlrPC, int32_t **excHndlrSP, int32_t **excHndlrFP) {
   static int wrapAroundSignald = 0;
   switch (procNo) {
      case 0: {      // PROCEDURE getResolution(): INTEGER 
         (*SP)--;
         **SP = 1000;
         break;
      }
      case 1: {      // PROCEDURE now(): TimeStamp
         int n = now();
         (*SP)--;
         **SP = n;
	 if (n > ALARM_THRESHOLD && !wrapAroundSignald && wrapAroundTask != NULL) stdWakeup(wrapAroundTask);
         wrapAroundSignald = 1;
         break;
      }
      case 2: {      // PROCEDURE reset()
         pthread_mutex_lock(&stamperMutex);
         lst_ts = 0x80000000;
         pthread_mutex_unlock(&stamperMutex);
         wrapAroundSignald = 0;
         break;
      }
   }
   return 1;
}

