#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <pthread.h>
#include <string.h>

#include "vm.h"
#include "tcpsocket.h"

struct socketSpec {
   int listenSock, dataSock, port, startListen, connUp;
   struct sockaddr_in addr;
   pthread_mutex_t Mutex;
   pthread_cond_t Cond;
   pthread_mutex_t writeMutex;
};

int initTcpSocket(ComponentInst *inst) {
   pthread_mutexattr_t mta;
   struct socketSpec *s = malloc(sizeof(struct socketSpec));
   if (s == NULL) return 0;
   if (pthread_mutex_init(&(s->Mutex), NULL)) fprintf(stderr, "initTCPsocket: pthread_mutex_init");
   if (pthread_cond_init(&(s->Cond), NULL)) fprintf(stderr, "initTCPsocket: pthread_cond_init");
   if (pthread_mutexattr_init(&mta)) fprintf(stderr, "initTCPsocket: pthread_mutexattr_init");      
   if (pthread_mutexattr_settype(&mta, PTHREAD_MUTEX_RECURSIVE)) fprintf(stderr, "initTCPsocket: pthread_mutexattr_settype");
   if (pthread_mutex_init(&(s->writeMutex), &mta)) fprintf(stderr, "initTCPsocket: pthread_mutex_init");
   s->dataSock = s->listenSock = s->port = -1;
   s->startListen = s->connUp = 0;
   inst->instSpecific = s;
   return 1;
}

static void ConnectionEstablishedTrigger(vm *task) {
   struct socketSpec *s = (struct socketSpec *)(task->triggerInst->instSpecific);
   struct sockaddr_in rem;   
   int slen, ds = -1, flag = 1;

   do {
      if (pthread_mutex_lock(&(s->Mutex))) fprintf(stderr, "TCPsocket, ConnectionEstablishedTrigger, pthread_mutex_lock");
      while (!s->startListen) {
         if (pthread_cond_wait(&(s->Cond), &(s->Mutex))) fprintf(stderr, "TCPsocket, ConnectionEstablishedTrigger, pthread_cond_wait");
      }
      pthread_mutex_unlock(&(s->Mutex));
 
      if (listen(s->listenSock, 1) < 0) fprintf(stderr, "TCPsocket, ConnectionEstablishedTrigger, listen");
      slen = sizeof(struct sockaddr_in); 
      if((ds = accept(s->listenSock, (struct sockaddr *)&rem, &slen)) < 0) fprintf(stderr, "TCPsocket, ConnectionEstablishedTrigger, accept");
   } while (ds < 0);

   pthread_mutex_lock(&(s->Mutex));
   s->dataSock = ds;
   setsockopt(ds, IPPROTO_TCP, TCP_NODELAY, (char *) &flag, sizeof(int));
   s->startListen = 0; s->connUp = 1;
   pthread_cond_broadcast(&(s->Cond));
   pthread_mutex_unlock(&(s->Mutex));
}

static void ConnectionBrokenTrigger(vm *task) {
   struct socketSpec *s = (struct socketSpec *)(task->triggerInst->instSpecific);
   if (pthread_mutex_lock(&(s->Mutex))) fprintf(stderr, "TCPsocket, ConnectionBrokenTrigger, pthread_mutex_lock");
   for(;;) {
      if (s->dataSock == -1 && s->connUp) break;
      if (pthread_cond_wait(&(s->Cond), &(s->Mutex))) fprintf(stderr, "TCPsocket, ConnectionBrokenTrigger, pthread_cond_wait");
   }
   s->connUp = 0;
   pthread_mutex_unlock(&(s->Mutex));
}

static void DataAvailableTrigger(vm *task) {
   struct socketSpec *s = (struct socketSpec *)(task->triggerInst->instSpecific);
   int ds;
   fd_set rd, ex;

   do {
      if (pthread_mutex_lock(&(s->Mutex))) fprintf(stderr, "TCPsocket, DataAvailableTrigger, pthread_mutex_lock");
      while (s->dataSock < 0) {
         if (pthread_cond_wait(&(s->Cond), &(s->Mutex))) fprintf(stderr, "TCPsocket, DataAvailableTrigger, pthread_cond_wait");
      }
      ds = s->dataSock;
      pthread_mutex_unlock(&(s->Mutex));
      FD_ZERO(&rd); FD_ZERO(&ex);
      FD_SET(ds, &rd); FD_SET(ds, &ex);
      select(ds+1, &rd, NULL, &ex, NULL);
   } while (s->dataSock < 0 || !FD_ISSET(ds, &rd));
}

void armTcpSocket(vm *task) {
   int32_t eventNo = task->triggerIdx;
   ComponentInst *inst = task->triggerInst;
   if (inst->events[eventNo] != NULL) {
      if (eventNo == 0) {
         task->triggerFun = ConnectionEstablishedTrigger;
      } else if (eventNo == 1) {
         task->triggerFun = ConnectionBrokenTrigger;
      } else if (eventNo == 2) { 
         task->triggerFun = DataAvailableTrigger;
      }
   }
}

static int offerService(ComponentInst *inst, int port) {
   struct socketSpec *s = (struct socketSpec *)(inst->instSpecific);

   pthread_mutex_lock(&(s->Mutex));
   if (s->listenSock == -1) {
   	s->listenSock = socket(PF_INET, SOCK_STREAM, 0);
   }
   if (port != s->port) {
      s->addr.sin_family = AF_INET;
      s->addr.sin_port = htons(port);
      s->addr.sin_addr.s_addr = INADDR_ANY;
      if(bind(s->listenSock, (struct sockaddr *)&(s->addr), sizeof(struct sockaddr_in)) < 0) {
         pthread_mutex_unlock(&(s->Mutex));
         return 0;
      }
      s->port = port;
   }
   s->startListen = 1;
   pthread_cond_broadcast(&(s->Cond));
   pthread_mutex_unlock(&(s->Mutex));      

   return 1;
}

static int writeString(int so, char *s) {
   int c, l;

   if (so < 0) return -1;   
   c = strlen(s);
   l = send(so, s, c, 0);
   if (l < 0) return -1;
   if (l < c) return 0;
   return 1;
}

static int writeWord(int so, int32_t w) {
   int l;
   
   if (so < 0) return -1;
   l = send(so, &w, sizeof(int32_t), 0);
   if (l < 0) return -1;
   if (l < sizeof(int32_t)) return 0;
   return 1;
}

static int writeBlob(int so, BlobHeader *b) {
   int c, l;
   
   if (so < 0) return -1;
   c = (b->dataCnt + 2) * sizeof(int32_t);
   l = send(so, &b->tag, c, 0);
   if (l < 0) return -1;
   if (l < c) return 0;
   return 1;
}

static int readWord(int so, int32_t *w) {
   int l;
   
   if (so < 0) return -1;
   l = recv(so, w, sizeof(int32_t), MSG_WAITALL);
   if (l < 0) return -1;
   if ( l < sizeof(int32_t)) return 0;
   return 1;
}

static int readBlob(int so, BlobHeader **b) {
   int32_t tagNcnt[2], l, c;
   char *d;

   if (so < 0) return -1;
   l = recv(so, tagNcnt, 2 * sizeof(int32_t), MSG_WAITALL);
   if (l < 0) return -1;
   if (l < 2 * sizeof(int32_t)) return 0;
   // printf("tcpsocket.readBlob, allocating Blob, size = %d\n", tagNcnt[1]);
   if ((*b = allocBlob(tagNcnt[1])) == NULL) return -1;
   (*b)->tag = tagNcnt[0];
   d = (char *)((*b)->data);
   l = recv(so, d, tagNcnt[1] * sizeof(int32_t), MSG_WAITALL);
   if (l < 0) return -1;
   if (l < tagNcnt[1] * sizeof(int32_t)) return 0;
   return 1;
}

int doTcpSocket(int32_t procNo, ComponentInst *inst, int32_t **PC, int32_t **SP, int32_t **FP, int32_t *excSP, int32_t **excHndlrPC, int32_t **excHndlrSP, int32_t **excHndlrFP) {
   switch (procNo) {
      case 0: {         // PROCEDURE offerService(INTEGER port);
         int32_t port;
         port = **SP; (*SP)++;
         if (!offerService(inst, port)) return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x20, "error setting up socket");
         break;
      }
      case 1: {         // PROCEDURE close();
         struct socketSpec *s = (struct socketSpec *)(inst->instSpecific);
         int ds;
         pthread_mutex_lock(&s->Mutex);
         ds = s->dataSock;
         if (ds != -1) {
            s->dataSock = -1;
            close(ds);
            pthread_cond_broadcast(&(s->Cond));
         }
         pthread_mutex_unlock(&s->Mutex);
         break;
      }
      case 2: {         // PROCEDURE writeString(STRING Text);
         struct socketSpec *s = (struct socketSpec *)(inst->instSpecific);
         int32_t r;
         char *c = (char *)Memory[**SP]; (*SP)++;
         pthread_mutex_lock(&s->writeMutex);
         pthread_mutex_lock(&s->Mutex);
         r = writeString(s->dataSock, c);
         pthread_mutex_unlock(&s->Mutex);
         pthread_mutex_unlock(&s->writeMutex);
         if (r <= 0) {
            pthread_mutex_lock(&s->Mutex);
            close(s->dataSock); s->dataSock = -1;
            pthread_cond_broadcast(&(s->Cond));
            pthread_mutex_unlock(&s->Mutex);
            return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x20, (r == 0) ? "connection closed" : "error writing string to socket");
         }       
         break;
      }
      case 3:           // PROCEDURE writeInteger(INTEGER i);
      case 4:           // PROCEDURE writeFloat(FLOAT f);
      case 5:           // PROCEDURE writeBool(BOOL b);
      case 6:           // PROCEDURE writeTimeStamp(TIMESTAMP t);
      {
         struct socketSpec *s = (struct socketSpec *)(inst->instSpecific);
         int32_t d, r;
         d = **SP; (*SP)++;
         pthread_mutex_lock(&s->writeMutex);
         pthread_mutex_lock(&s->Mutex);
         r = writeWord(s->dataSock, d);
         pthread_mutex_unlock(&s->Mutex);
         pthread_mutex_unlock(&s->writeMutex);
         if (r <= 0) {
            pthread_mutex_lock(&s->Mutex);
            close(s->dataSock); s->dataSock = -1;
            pthread_cond_broadcast(&(s->Cond));
            pthread_mutex_unlock(&s->Mutex);
            return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x20, (r == 0) ? "connection closed" : "error writing data to socket");
         }       
         break;
      }
      case 7: {         // PROCEDURE writeBlob(BLOB b);
         struct socketSpec *s = (struct socketSpec *)(inst->instSpecific);
         int32_t r;
         BlobHeader *b;
         b = (BlobHeader *)(**SP);
         if (b != NULL) {
            pthread_mutex_lock(&s->writeMutex);
            pthread_mutex_lock(&s->Mutex);
            r = writeBlob(s->dataSock, b);
            pthread_mutex_unlock(&s->Mutex);
            pthread_mutex_unlock(&s->writeMutex);

            WRITELOCK;
            dwnRefCnt(*SP);
            UNLOCK;
         }
         (*SP)++;
         if (r <= 0) {
            pthread_mutex_lock(&s->Mutex);
            close(s->dataSock); s->dataSock = -1;
            pthread_cond_broadcast(&(s->Cond));
            pthread_mutex_unlock(&s->Mutex);
            return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x20, (r == 0) ? "connection closed" : "error writing blob to socket");
         }       
         break;
      }
      case 8: {         // PROCEDURE startSequence();
         struct socketSpec *s = (struct socketSpec *)(inst->instSpecific);
         pthread_mutex_lock(&s->writeMutex);
         break;
      }
      case 9: {         // PROCEDURE endSequence();
         struct socketSpec *s = (struct socketSpec *)(inst->instSpecific);
         pthread_mutex_unlock(&s->writeMutex);
         break;
      }
      case 10:          // PROCEDURE readInteger(): INTEGER;
      case 11:          // PROCEDURE readFloat(): FLOAT;
      case 12:          // PROCEDURE readBool(): BOOL;
      case 13:          // PROCEDURE readTimeStamp(): TIMESTAMP;
      {
         struct socketSpec *s = (struct socketSpec *)(inst->instSpecific);
         int32_t d, r;
         pthread_mutex_lock(&s->Mutex);
         r = readWord(s->dataSock, &d);
         if (r <= 0) {
            close(s->dataSock); s->dataSock = -1;
            pthread_cond_broadcast(&(s->Cond));
            pthread_mutex_unlock(&s->Mutex);
            return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x20, (r == 0) ? "connection closed" : "error reading data from socket");
         }       
         pthread_mutex_unlock(&s->Mutex);
         (*SP)--;
         **SP = d;
         break;
      }
      case 14: {        // PROCEDURE readBlob(): BLOB
         struct socketSpec *s = (struct socketSpec *)(inst->instSpecific);
         int32_t r;
         BlobHeader *b;
         pthread_mutex_lock(&s->Mutex);
         r = readBlob(s->dataSock, &b);
         if (r <= 0) {
            close(s->dataSock); s->dataSock = -1;
            pthread_cond_broadcast(&(s->Cond));
            pthread_mutex_unlock(&s->Mutex);
            return doThrow(PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x20, (r == 0) ? "connection closed" : "error reading blob from socket");
         }
         pthread_mutex_unlock(&s->Mutex);
         (*SP)--;
         **SP = (int32_t)b;
         break;
      }
   }
   return 1;
}

