#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <math.h>
#include <limits.h>
#include <mcheck.h>

#include <pthread.h>
#include <sched.h>
#include <signal.h>
#include <time.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>

#include "vm.h"

#define LITTLE

int32_t majorvers = 1, minorvers = 17;

pthread_rwlock_t rw_global;

pthread_mutex_t BlobMemMutex;

pthread_attr_t vmAttrs;

#define DATA_SIZE 0x100000 
#define RAM_WIDTH sizeof(int32_t)

int32_t Memory[DATA_SIZE] = {0}, *Code, staticEnd, constBegin, TskNo, stringVars;
vm *Tasks;
ComponentDef *compDefs;
ComponentInst *compInsts;
ComponentVar *compVars;
int compVarCnt;

static int32_t loglevel = 0;


#ifdef LITTLE
static void swapBytes(int32_t *d) {
   char *cd = (char *)d, t;
   t = cd[0]; cd[0] = cd[3]; cd[3] = t;
   t = cd[1]; cd[1] = cd[2]; cd[2] = t;
}
#endif

static int32_t readInt(FILE *f) {
   int32_t d;
   int s1 = fread(&d, sizeof(int32_t), 1, f);
#ifdef LITTLE
   swapBytes(&d);
#endif          
   return d;
}

static void readIntArray(FILE *f, int32_t d[], int32_t l) {
   int32_t i;
   fread(d, sizeof(int32_t), l, f);
#ifdef LITTLE
   for (i = 0; i < l; i++) {
      char *cd = (char *)(d + i), t;
      t = cd[0]; cd[0] = cd[3]; cd[3] = t;
      t = cd[1]; cd[1] = cd[2]; cd[2] = t;
   }        
#endif
}

static TypeDef readTypeDef(FILE *f) {
   TypeDef td;
   td.tag = readInt(f);
   if (td.tag == NOTYPE || (BOOL <= td.tag && td.tag <= BLOB)) {
      td.par = 0; td.subs = NULL;
   } else if (td.tag == ARRAY) {
      td.par = readInt(f);
      td.subs = malloc(sizeof(TypeDef));
      *(td.subs) = readTypeDef(f);
   } else if (td.tag == RECORD || td.tag == TUPPLE) {
      int32_t i;
      td.par = readInt(f);
      td.subs = calloc(sizeof(TypeDef), td.par);
      for (i = 0; i < td.par; i++) td.subs[i] = readTypeDef(f);
   }
   return td;
}

static ProcInterfaceDef readProcInterfaceDef(FILE *f) {
   ProcInterfaceDef id;
   int32_t i;

   id.tag = readInt(f);
   id.parcnt = readInt(f);
   id.params = calloc(sizeof(TypeDef), id.parcnt);
   for (i = 0; i < id.parcnt; i++) {
      id.params[i] = readTypeDef(f); 
   }
   id.ret = readTypeDef(f);
   return id;   
}

static vm readBinary(FILE *f) {
   int32_t codeSize, s, i, j, a, compCnt;
   vm r;

   // read code   
   codeSize = readInt(f);
   Code = calloc(sizeof(int32_t), codeSize);
   readIntArray(f, Code, codeSize);
   // read Component Definitions
   compCnt = readInt(f);
   compDefs = calloc(sizeof(ComponentDef), compCnt);
   for (i = 0; i < compCnt; i++) {
      int len = readInt(f);
      char *name = malloc(len + 1);
      fread(name, sizeof(char), len, f);
      name[len] = '\0';
      // printf("reading component definition %s\n", name);
      compDefs[i].name = name;
      compDefs[i].CompProcCnt = readInt(f);      
      compDefs[i].interfaces = calloc(sizeof(ProcInterfaceDef), compDefs[i].CompProcCnt);
      for (j = 0; j < compDefs[i].CompProcCnt; j++) {
         compDefs[i].interfaces[j] = readProcInterfaceDef(f);
      } 
      if (!initCompDef(compDefs + i)) {
         fprintf(stderr, "unknown COMPONENT type %s\n", compDefs[i].name);
         exit(EXIT_FAILURE);
      }
   }
   // read Component Instances
   compCnt = readInt(f);
   compInsts = calloc(sizeof(ComponentInst), compCnt);
   for (i = 0; i < compCnt; i++) {
      int offset = readInt(f);
      compInsts[i].objDict = Memory + offset + 1;
      int len = readInt(f);
      int evCnt;
      char *name = malloc(len + 1);
      fread(name, sizeof(char), len, f); name[len] = '\0';
      // printf("initializing instance %s, offset = %d\n", name, offset);
      compInsts[i].instName = name;
      len = readInt(f);
      name = malloc(len + 1);
      fread(name, sizeof(char), len, f); name[len] = '\0';
      compInsts[i].defName = name;
      evCnt = readInt(f);
      compInsts[i].events = (evCnt == 0) ? NULL : calloc(sizeof(vm *), evCnt);
      Memory[offset] = (int32_t)(compInsts + i);
   }
   // read taks
   TskNo = readInt(f);
   Tasks = calloc(sizeof(vm), TskNo);
   for (i = 0; i < TskNo; i++) {
      int32_t c, CompOff, EvIdx;
      c = readInt(f);
      Tasks[i].name = malloc(c + 1); Tasks[i].name[c] = '\0';
      fread(Tasks[i].name, sizeof(char), c, f);
      Tasks[i].pc = Code + readInt(f);
      Tasks[i].period = readInt(f);
      Tasks[i].statics = Memory + readInt(f);
      CompOff = readInt(f);
      EvIdx = readInt(f);
      if (CompOff != -1) {
         ComponentInst *inst = (ComponentInst *)(Memory[CompOff]);
         inst->events[EvIdx] = Tasks + i;
         Tasks[i].triggerInst = inst;
         Tasks[i].triggerIdx = EvIdx;
      } else {
         Tasks[i].triggerInst = NULL;
      }     
   }
   // Area for stacks
   staticEnd = readInt(f); constBegin = readInt(f);
   r.sp = Memory + (constBegin + staticEnd) / 2  - 1;
   r.pc = Code;

   // Component instances initialization
   stringVars = constBegin - 1;
   for (i = 0; i < compCnt; i++) {
      if (!initCompInst(compInsts + i, compDefs)) {
         fprintf(stderr, "unknown COMPONENT instance, name of component %s, name of instance %s\n", (compInsts + i)->defName, (compInsts + i)->instName);
         exit(EXIT_FAILURE);
      }
   }

   // integers
   s = readInt(f);   
   for (i = 0; i < s; i++) {
      a = readInt(f);
      Memory[a] = readInt(f);
   }
   // floats
   s = readInt(f);   
   for (i = 0; i < s; i++) {
      a = readInt(f);
      Memory[a] = readInt(f);
   }
   // blobLists
   s = readInt(f);   
   for (i = 0; i < s; i++) {
      int32_t c, j;
      a = readInt(f);
      c = readInt(f);
      for (j = 0; j < c; j++) Memory[a + j] = readInt(f);
   }
   // strings
   s = readInt(f);
   for (i = 0; i < s; i++) {
      int32_t c;
      char *b;
      a = readInt(f);
      c = readInt(f);
      b = malloc(c);
      Memory[a] = (int32_t)(b + 1);
      fread(b, sizeof(char), c, f);
      b[c - 1] = '\0';
   }
   compVarCnt = readInt(f);
   compVars = malloc(compVarCnt * sizeof(ComponentVar));
   for (i = 0; i < compVarCnt; i++) {
      int32_t c;
      c = readInt(f);
      compVars[i].name = malloc(c + 1); compVars[i].name[c] = '\0';
      fread(compVars[i].name, sizeof(char), c, f);
      compVars[i].ty = readTypeDef(f);
      compVars[i].addr = readInt(f); compVars[i].eaddr = readInt(f);
   }
   return r;
}

void releaseParameter(TypeDef *ty, int32_t **SP) {
   // printf("in releaseParameter, ");
   if (ty->tag == BOOL || ty->tag == INTEGER || ty->tag == FLOAT || ty->tag == STRING || ty->tag == TIMESTAMP) {
      // printf("releasing scalar\n");
      (*SP)++;
   } else if (ty->tag == BLOB) {
      // printf("releasing blob, SP = %x, ", *SP);
      WRITELOCK; dwnRefCnt(*SP); UNLOCK; (*SP)++;
      // printf("done releasing, SP = %x\n", *SP);
   } else if (ty->tag == ARRAY) {
      // printf("releasing array\n");
      if (ty->subs->tag == BOOL || ty->subs->tag == INTEGER || ty->subs->tag == FLOAT || ty->subs->tag == STRING || ty->subs->tag == TIMESTAMP) (*SP)+= ty->par;
      else {
         int32_t i;
         for (i = 0; i < ty->par; i++) releaseParameter(ty->subs, SP);
      }
   } else if (ty->tag == RECORD) {
      int32_t i;
      // printf("releasing record\n");
      for (i = 0; i < ty->par; i++) releaseParameter(ty->subs + i, SP);
   }
}

void logError(int level, char *reason) {
   if (level >= loglevel) {
      time_t t = time(NULL);
      char ts[26];
      struct tm tt;
      asctime_r(gmtime_r(&t, &tt), ts);
      ts[24] = '\0'; 
      fprintf(stderr, "%s, %s\n", ts, reason);
   }
}

int doThrow(int32_t **PC, int32_t **SP, int32_t **FP, int32_t *excSP, int32_t **excHndlrPC, int32_t **excHndlrSP, int32_t **excHndlrFP, int level, char *reason) {
   // printf("doThrow, PC = %d\n", *PC - Code);
   logError(level, reason);
   *PC = excHndlrPC[*excSP]; *SP = excHndlrSP[*excSP]; 
   while (*FP != excHndlrFP[*excSP]) {
      int32_t blobLst = BLOB_LST(Code[*(*FP - 1)]), i;
      // printf("in doThrow, blobLst = %d\n", blobLst);
      if (blobLst > 0) {
         WRITELOCK;
         for (i = 1; i <= blobLst; i++) dwnRefCnt(*FP + Code[*(*FP - 1) + i]);
         UNLOCK;
      }
      *FP = Memory + **FP;
   }
   (*excSP)--;
   if (*excSP == -1) return 0;
   return 1;
}

static int doCompCall(int32_t procNo, int32_t **PC, int32_t **SP, int32_t **FP, int32_t *excSP, int32_t **excHndlrPC, int32_t **excHndlrSP, int32_t **excHndlrFP) {
   ComponentInst *inst = (ComponentInst *)(**SP);
   (*SP)++;
   return inst->definition->CompProc(procNo, inst, PC, SP, FP, excSP, excHndlrPC, excHndlrSP, excHndlrFP);
}

void doFetchDecodeExecuteLoop(vm* v) {
   int32_t *PC = v->pc, *SP = v->sp, *FP = v->sp, running = 1;
   int32_t excSP = 0, *excHndlrPC[10], *excHndlrSP[10], *excHndlrFP[10];
   excHndlrPC[0] = Code; excHndlrSP[0] = excHndlrFP[0] = v->sp; 
   #define PUSH(v) do {*(--SP) = v;} while(0)
   #define PUSH_FLOAT(v) do {*((float *)(--SP)) = v;} while(0)
   #define POP (*SP++)
   #define POP_FLOAT (*((float *)(SP++)))
   #define PEEK(o) (*(SP + o))

   while (running) {
      int32_t inst = *PC;

      switch((inst >> 24) & 0xFF) {
         case LOAD: {
            int32_t address = ADDRESS(inst);
            if (IS_BLOB(inst)) {WRITELOCK; upRefCnt(Memory + address);} 
            PUSH(Memory[address]);
            if (IS_BLOB(inst)) UNLOCK;
            PC++;
            break;
         }
         case LOADA: {
            int32_t index = POP, size = *(PC + 1), address;
            if (0 <= index && index < size) {
               address = ADDRESS(inst) + index;
               if (IS_BLOB(inst)) {WRITELOCK; upRefCnt(Memory + address);} 
               PUSH(Memory[address]);
               if (IS_BLOB(inst)) UNLOCK;
               PC += 2;
            } else running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0xff, "index out of bounds");
            break;
         }
         case LOADL: {
            int32_t offset = OFFSET(inst);
            if (IS_BLOB(inst)) {WRITELOCK; upRefCnt(FP + offset);} 
            PUSH(*(FP + offset));
            if (IS_BLOB(inst)) UNLOCK;
            PC++;
            break;
         }
         case LOADAL: {
            int32_t index = POP, size = *(PC + 1), offset;
            if (0 <= index && index < size) {
               offset = OFFSET(inst) + index;
               if (IS_BLOB(inst)) {WRITELOCK; upRefCnt(FP + offset);} 
               PUSH(*(FP + offset));
               if (IS_BLOB(inst)) UNLOCK;
               PC += 2;
            } else running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0xff, "index out of bounds");
            break;
         }
         case LOADV: {
            int32_t address = ADDRESS(POP);
            if (IS_BLOB(inst)) {WRITELOCK; upRefCnt(Memory + address);} 
            PUSH(Memory[address]);
            if (IS_BLOB(inst)) UNLOCK;
            PC++;
            break;
         }
         case LOADM: {
            int32_t address, size = SIZE(*(PC + 1)), blobLst = BLOB_LST(*(PC + 1)), i;
            int32_t *src, *dst;
            if (IS_LOCAL(inst)) address = (FP - Memory) + OFFSET(inst);
            else address = ADDRESS(inst);
            SP -= size;
            if (blobLst > 0) WRITELOCK;
            else if (IS_LOCKED(inst)) READLOCK;
            PC += 2;
            for (i = 0; i < blobLst; i++) upRefCnt(Memory + address + *PC++);
            for (i = 0, dst = SP, src = Memory + address; i < size; i++, dst++, src++) *dst = *src;
            if (blobLst > 0 || IS_LOCKED(inst)) UNLOCK;
            break;
         } 
         case LOADMV: {
            int32_t address = POP, size = SIZE(inst), blobLst = BLOB_LST(inst), i;
            int32_t *src, *dst;
            SP -= size;
            if (blobLst > 0) WRITELOCK;
            else if (IS_ADDR_LOCKED(address)) READLOCK;
            PC++;
            for (i = 0; i < blobLst; i++) upRefCnt(Memory + address + *PC++);
            for (i = 0, dst = SP, src = Memory + ADDRESS(address); i < size; i++, dst++, src++) *dst = *src;
            if (blobLst > 0 || IS_ADDR_LOCKED(address)) UNLOCK;
            break;
         } 
         case STORE: {
            int32_t address = ADDRESS(inst), value = POP;
            if (IS_BLOB(inst)) {WRITELOCK; dwnRefCnt(Memory + address);} 
            Memory[address] = value;
            if (IS_BLOB(inst)) UNLOCK;
            PC++;
            break;
         }
         case STOREA: {
            int32_t value = POP, index = POP, size = *(PC + 1), address;
            if (0 <= index && index < size) {
               address = ADDRESS(inst) + index;
               if (IS_BLOB(inst)) {WRITELOCK; dwnRefCnt(Memory + address);} 
               Memory[address] = value;
               if (IS_BLOB(inst)) UNLOCK;
               PC += 2;
            } else {
               if (IS_BLOB(inst)) {
                  WRITELOCK; dwnRefCnt(&value); UNLOCK;
               }
               running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0xff, "index out of bounds");
            }
            break;
         }
         case STOREL: {
            int32_t value = POP, offset = OFFSET(inst);
            if (IS_BLOB(inst)) {WRITELOCK; dwnRefCnt(FP + offset);} 
            *(FP + offset) = value;
            if (IS_BLOB(inst)) UNLOCK;
            PC++;
            break;
         }
         case STOREAL: {
            int32_t value = POP, index = POP, size = *(PC + 1), offset;
            if (0 <= index && index < size) {
               offset = OFFSET(inst) + index;
               if (IS_BLOB(inst)) {WRITELOCK; dwnRefCnt(FP + offset);} 
               *(FP + offset) = value;
               if (IS_BLOB(inst)) UNLOCK;
               PC += 2;
            } else {
               if (IS_BLOB(inst)) {
                  WRITELOCK; dwnRefCnt(&value); UNLOCK;
               }
               running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0xff, "index out of bounds");
            }
            break;
         }
         case STOREM: {
            int32_t address, size = SIZE(*(PC + 1)), blobLst = BLOB_LST(*(PC + 1)), i;
            int32_t *src, *dst;
            if (IS_LOCAL(inst)) address = (FP - Memory) + OFFSET(inst);
            else address = ADDRESS(inst);
            if ((blobLst > 0) || (IS_LOCKED(inst))) WRITELOCK;
            PC += 2;
            for (i = 0; i < blobLst; i++) dwnRefCnt(Memory + address + *PC++);
            for (i = 0, src = SP, dst = Memory + address; i < size; i++, dst++, src++) *dst = *src;
            if (blobLst > 0 || IS_LOCKED(inst)) UNLOCK;
            SP += size;
            break;
         } 
         case STOREMV: {
            int32_t size = SIZE(inst), address = PEEK(size), blobLst = BLOB_LST(inst), i;
            int32_t *src, *dst;
            if ((blobLst > 0) || (IS_ADDR_LOCKED(address))) WRITELOCK;
            PC++;
            for (i = 0; i < blobLst; i++) dwnRefCnt(Memory + address + *PC++);
            for (i = 0, src = SP, dst = Memory + ADDRESS(address); i < size; i++, dst++, src++) *dst = *src;
            if (blobLst > 0 || IS_ADDR_LOCKED(address)) UNLOCK;
            SP += size + 1;
            break;
         } 
         case STOREP: {
            int32_t *lhs = SP + LEFT(inst) + RIGHT(inst), *rhs = SP + RIGHT(inst), *rhsst = rhs;
            SP = lhs;
            while (lhs > rhsst) {
               int32_t a = *(lhs - 1), m = *(lhs - 2), blobLst = BLOB_LST(m), size = SIZE(m), i, addr;
               addr = ADDRESS(a);
               if (IS_ADDR_LOCKED(a) || blobLst > 0) WRITELOCK;
               lhs -= 2;
               for (i = 0; i < blobLst; i++) dwnRefCnt(Memory + addr + *(--lhs));
               rhs -= size;
               for (i = 0; i < size; i++) Memory[addr++] = *rhs++;
               rhs -= size;
               if (IS_ADDR_LOCKED(a) || blobLst > 0) UNLOCK;               
            }
            PC++;
            break;
         }
         case ADDFP: {
            PUSH(FP - Memory + OFFSET(inst));
            PC++;
            break;
         }
         case CHECK: {
            int32_t index = PEEK(0);
            if (index < 0 || SIZE(inst) <= index)
               running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0xff, "index out of bounds");
            else PC++;
            break;
         }
         case XTRACT: {
            int32_t offset = OFFSET(inst), blobLst = BLOB_LST(inst), baseSiz = BASE_SIZE(*(PC + 1)), fldSiz = FIELD_SIZE(*(PC + 1)), i;
            int32_t *src, *dest;
            if (blobLst > 0) WRITELOCK;
            for (i = 0; i < blobLst; i++) {
               int32_t bo = *(PC + 2 + i);
               if (!(offset <= bo && bo < offset + fldSiz)) dwnRefCnt(SP + bo);
            }
            src = SP + offset + fldSiz - 1; dest = SP + baseSiz - 1; 
            for (i = fldSiz - 1; i >= 0; i--) *dest-- = *src--;
            if (blobLst > 0) UNLOCK;
            SP += (baseSiz - fldSiz);
            PC += (2 + blobLst);
            break;
         }
         case STKAREF: {
            int32_t offset = POP, blobLst = BLOB_LST(inst), baseSiz = BASE_SIZE(*(PC + 1)), fldSiz = FIELD_SIZE(*(PC + 1)), i;
            int32_t *src, *dest;
            if (blobLst > 0) WRITELOCK;
            for (i = 0; i < blobLst; i++) {
               int32_t bo = *(PC + 2 + i);
               if (!(offset <= bo && bo < offset + fldSiz)) dwnRefCnt(SP + bo);
            }
            src = SP + offset + fldSiz - 1; dest = SP + baseSiz - 1; 
            for (i = fldSiz - 1; i >= 0; i--) *dest-- = *src--;
            if (blobLst > 0) UNLOCK;
            SP += (baseSiz - fldSiz);
            PC += (2 + blobLst);
            break;
         }
         case ADD: {
            int32_t a = POP, b = POP;
            PUSH(a + b);
            PC++;
            break;
         }
         case SUB: {
            int32_t a = POP, b = POP;
            PUSH(b - a);
            PC++;
            break;
         }
         case MUL: {
            int32_t a = POP, b = POP;
            PUSH(a * b);
            PC++;
            break;
         }
         case DIV: {
            int32_t a = POP, b = POP;
            if (a == 0) {
               running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x1, "integer division by 0");
               break;
            }
            PUSH(b / a);
            PC++;
            break;
         }
         case MOD: {
            int32_t a = POP, b = POP;
            if (a == 0) {
               running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x1, "integer division by 0");
               break;
            }
            PUSH(b % a);
            PC++;
            break;
         }
         case NEG: {
            int32_t a = POP;
            PUSH(-a);
            PC++;
            break;
         }
         case AND: {
            int32_t a = POP, b = POP;
            PUSH(a & b);
            PC++;
            break;
         }
         case OR: {
            int32_t a = POP, b = POP;
            PUSH(a | b);
            PC++;
            break;
         }
         case XOR: {
            int32_t a = POP, b = POP;
            PUSH(a ^ b);
            PC++;
            break;
         }
         case NOT: {
            int32_t a = POP;
            PUSH(~a);         
            PC++;
            break;
         }
         case DURATION: {
            int32_t a = POP, b = POP;
            a = a - b;
            if (a < 0) a = -a;
            a++;
            PUSH(a);
            PC++;
            break;
         }
         case EARLIEST: {
            int32_t c = SIZE(inst), i, a = POP;
            for (i = 1; i < c; i++) {
               int32_t b = POP;
               if (b < a) a = b;
            }   
            PUSH(a);
            PC++;
            break;
         }
         case LATEST: {
            int32_t c = SIZE(inst), i, a = POP;
            for (i = 1; i < c; i++) {
               int32_t b = POP;
               if (b > a) a = b;
            }   
            PUSH(a);
            PC++;
            break;
         }
         case LT: {
            int32_t a = POP, b = POP;
            PUSH(b < a);
            PC++;
            break;
         }
         case LE: {
            int32_t a = POP, b = POP;
            PUSH(b <= a);
            PC++;
            break;
         }
         case EQ: {
            int32_t a = POP, b = POP;
            PUSH(b == a);
            PC++;
            break;
         }
         case GE: {
            int32_t a = POP, b = POP;
            PUSH(b >= a);
            PC++;
            break;
         }
         case GT: {
            int32_t a = POP, b = POP;
            PUSH(b > a);
            PC++;
            break;
         }
         case NEQ: {
            int32_t a = POP, b = POP;
            PUSH(b != a);
            PC++;
            break;
         }
         case FADD: {
            float a = POP_FLOAT, b = POP_FLOAT;
            if (isnan(a) || isnan(b)) {
               running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x0, "floating point NaN encountered");
               break;
            }
            PUSH_FLOAT(a + b);
            PC++;
            break;
         }
         case FSUB: {
            float a = POP_FLOAT, b = POP_FLOAT;
            if (isnan(a) || isnan(b)) {
               running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x0, "floating point NaN encountered");
               break;
            }
            PUSH_FLOAT(b - a);
            PC++;
            break;
         }
         case FMUL: {
            float a = POP_FLOAT, b = POP_FLOAT;
            if (isnan(a) || isnan(b)) {
               running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x0, "floating point NaN encountered");
               break;
            }
            PUSH_FLOAT(a * b);
            PC++;
            break;
         }
         case FDIV: {
            float a = POP_FLOAT, b = POP_FLOAT;
            if (isnan(a) || isnan(b)) {
               running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x0, "floating point NaN encountered");
               break;
            }
            if (a == 0.0F) {
               running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x0, "floating point division by 0.0");
               break;
            }
            PUSH_FLOAT(b / a);
            PC++;
            break;
         }
         case FNEG: {
            float a = POP_FLOAT;
            if (isnan(a)) {
               running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x0, "floating point NaN encountered");
               break;
            }
            PUSH_FLOAT(-a);
            PC++;
            break;
         }
         case FLT: {
            float a = POP_FLOAT, b = POP_FLOAT;
            if (isnan(a) || isnan(b)) {
               running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x0, "floating point NaN encountered");
               break;
            }
            PUSH(b < a);
            PC++;
            break;
         }
         case FLE: {
            float a = POP_FLOAT, b = POP_FLOAT;
            if (isnan(a) || isnan(b)) {
               running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x0, "floating point NaN encountered");
               break;
            }
            PUSH(b <= a);
            PC++;
            break;
         }
         case FEQ: {
            float a = POP_FLOAT, b = POP_FLOAT;
            if (isnan(a) || isnan(b)) {
               running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x0, "floating point NaN encountered");
               break;
            }
            PUSH(b == a);
            PC++;
            break;
         }
         case FGE: {
            float a = POP_FLOAT, b = POP_FLOAT;
            if (isnan(a) || isnan(b)) {
               running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x0, "floating point NaN encountered");
               break;
            }
            PUSH(b >= a);
            PC++;
            break;
         }
         case FGT: {
            float a = POP_FLOAT, b = POP_FLOAT;
            if (isnan(a) || isnan(b)) {
               running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x0, "floating point NaN encountered");
               break;
            }
            PUSH(b > a);
            PC++;
            break;
         }
         case FNEQ: {
            float a = POP_FLOAT, b = POP_FLOAT;
            if (isnan(a) || isnan(b)) {
               running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x0, "floating point NaN encountered");
               break;
            }
            PUSH(b != a);
            PC++;
            break;
         }
         case TOFLOAT: {
            int32_t a = POP;
            PUSH_FLOAT((float) a);
            PC++;
            break;
         }
         case TOINT: {
            float a = POP_FLOAT;
            if (isnan(a)) {
               running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x0, "floating point NaN encountered");
               break;
            }
            if (a < INT32_MIN || INT32_MAX < a) {
               running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0x1, "overflow in FLOAT to INTEGER conversion");
               break;
            }
            PUSH((int) a);
            PC++;
            break;
         }
         case JMP: {
            PC += TARGET(inst);
            break;
         }
         case ANDJMP: {
            int32_t c = PEEK(0);
            if (!(c & 0x1)) PC += TARGET(inst);
            else {
               SP++;
               PC++;
            }
            break;
         }
         case ORJMP: {
            int32_t c = PEEK(0);
            if (c & 0x1) PC += TARGET(inst);
            else {
               SP++;
               PC++;
            }
            break;
         }
         case FALSEJMP: {
            int32_t c = POP;
            if (!(c & 0x1)) PC += TARGET(inst);
            else PC++;
            break;
         }
         case TRUEJMP: {
            int32_t c = POP;
            if (!c) PC += TARGET(inst);
            else PC++;
            break;
         }
         case CALL: {
            PUSH(PC - Code + 1);
            PC += TARGET(inst); 
            break;
         }
         case ENTER: {
            int32_t *l, blobLst = BLOB_LST(inst);
            PUSH(FP - Memory);
            FP = SP;
            PUSH(PC - Code);         // to find Bloblist
            SP -= SIZE(inst);        // allocate local variables
            for (l = SP; l < FP - 1; l++) *l = 0;
            PC += (blobLst + 1); 
            break;
         }
         case RETURN: {
            int32_t parSize = SIZE(inst), blobLst, i;
            blobLst = BLOB_LST(Code[*(FP - 1)]);
            if (blobLst > 0) {
               WRITELOCK;
               for (i = 1; i <= blobLst; i++) dwnRefCnt(FP + Code[*(FP - 1) + i]);
               UNLOCK;
            }
            SP = FP; 
            FP = Memory + POP;
            PC = Code + POP;
            SP += parSize;            
            break;
         }
         case FRETURN: {
            int32_t parSize = SIZE(inst), blobLst, retSize = SIZE(*(PC + 1)), i, *t;
            blobLst = BLOB_LST(Code[*(FP - 1)]);
            if (blobLst > 0) {
               WRITELOCK;
               for (i = 1; i <= blobLst; i++) dwnRefCnt(FP + Code[*(FP - 1) + i]);
               UNLOCK;
            }
            t = SP + retSize;
            SP = FP; 
            FP = Memory + POP;
            PC = Code + POP;
            SP += parSize;            
            for (i = 0; i < retSize; i++) *(--SP) = *(--t);
            break;
         }
         case COMPCALL: {
            int32_t procNo = SIZE(inst);
            PC ++;
            running = doCompCall(procNo, &PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP);
            break; 
         }
         case UPFORTST: {
            if (*SP > *(SP + 2)) {
               SP += 3; PC += TARGET(inst);
            } else PC++;
            break;
         }
         case DWNFORTST: {
            if (*SP < *(SP + 2)) {
               SP += 3; PC += TARGET(inst);
            } else PC++;
            break;
         }
         case FOREND: {
            *SP += *(SP + 1);
            PC += TARGET(inst);
            break;
         }
         case TRY: {
            excSP++;
            excHndlrPC[excSP] = PC + TARGET(inst);
            excHndlrSP[excSP] = SP;
            excHndlrFP[excSP] = FP;
            PC++;
            break;
         }
         case THROW: {
            running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0xff, (char *)(Memory[ADDRESS(inst)]));
            break;
         }
         case TRYEND: {
            excSP--;
            PC += TARGET(inst);
            break;
         }
         case STOP: {
            running = 0;
            break;
         }
         default: {
            fprintf(stderr, "PC = %d, opcode = %8x\n", PC - Code, (int)inst);
            running = doThrow(&PC, &SP, &FP, &excSP, excHndlrPC, excHndlrSP, excHndlrFP, 0xff, "Illegal Instruction");
            break;
         }         
      }
   }
}

struct Bin {
   int32_t dataCnt;
   BlobHeader *Blobs;
};

static int binCnt = 0, mxBinCnt;

static struct Bin *bins = NULL; 

static void initBlobMem(void) {
   if (pthread_mutex_init(&BlobMemMutex, NULL)) {
      fprintf(stderr, "BlobMemMutex: pthread_mutex_init");
      exit(EXIT_FAILURE);
   }
   bins = realloc(NULL, 10 * sizeof(struct Bin));
   mxBinCnt = 10;
}

BlobHeader *allocBlob(int dataSize) {
   BlobHeader *blHd = NULL;
   int i, j;
   if (pthread_mutex_lock(&BlobMemMutex)) {
      fprintf(stderr, "BlobMemMutex: pthread_mutex_lock");
      exit(EXIT_FAILURE);
   }
   i = 0;
   for (;;) {
      if (i == binCnt) {
          if (binCnt == mxBinCnt) {
             mxBinCnt += 10;
             bins = realloc(bins, mxBinCnt * sizeof(struct Bin));
             if (bins == NULL) {
                fprintf(stderr, "cannot allocate BLOB\n");
                exit(EXIT_FAILURE);
             } 
          }
          bins[i].dataCnt = dataSize;
          bins[i].Blobs = NULL;
          binCnt++;
          break;
      }
      if (bins[i].dataCnt == dataSize) break;
      i++;
   }
   if (bins[i].Blobs == NULL) {
//      printf("allocating BLOB, size = %d\n", dataSize);
      blHd = malloc(sizeof(BlobHeader) + dataSize * sizeof(int32_t));
      if (blHd == NULL) {
         fprintf(stderr, "cannot allocate BLOB\n");
         exit(EXIT_FAILURE);
      }
   } else {
      blHd = bins[i].Blobs;
      bins[i].Blobs = (BlobHeader *)(blHd->refCnt); 
   }
   // printf("allocated BLOB = 0x%x\n", (unsigned)blHd);
   pthread_mutex_unlock(&BlobMemMutex);
   blHd->refCnt = 1; blHd->dataCnt = dataSize;
   return blHd;
}

void freeBlob(BlobHeader *b) {
   int i;
   if (pthread_mutex_lock(&BlobMemMutex)) {
      fprintf(stderr, "BlobMemMutex: pthread_mutex_lock\n");
      exit(EXIT_FAILURE);
   }
   // printf("freed BLOB = 0x%x\n", (unsigned)b);
   i = 0;
   for (;;) {
      if (i == binCnt) {
           fprintf(stderr, "cannot free BLOB\n");
           exit(EXIT_FAILURE);
      } 
      if (bins[i].dataCnt == b->dataCnt) break;
      i++;
   }
   b->refCnt = (int32_t)(bins[i].Blobs);
   bins[i].Blobs = b;
   pthread_mutex_unlock(&BlobMemMutex);
}

static void wakeupPeriodicTask(union sigval p) {
   vm *task = (vm *)(p.sival_ptr);
   int32_t r;

   pthread_mutex_lock(&(task->mutex));
   if (!(r = task->running)) {
      task->running = 1;      
      pthread_cond_signal(&(task->cond));
   }
   pthread_mutex_unlock(&(task->mutex));
   if (r) {
      char reason[100];
      sprintf(reason, "realtime violation in task %20s", task->name);
      logError(0xff, reason);
   }
}

void stdWakeup(vm *task) {
   pthread_mutex_lock(&(task->mutex));
   if (!task->running) {
      task->running = 1;      
      pthread_cond_signal(&(task->cond));
   }
   pthread_mutex_unlock(&(task->mutex));
}

void stdTriggerFun(vm *task) {
   if (pthread_mutex_lock(&(task->mutex))) {
      fprintf(stderr, "stdTriggerFun, pthread_mutex_lock"); return;
   }
   task->running = 0;
   // printf("task %s waiting\n", task->name);
   while(!task->running) {
      if (pthread_cond_wait(&(task->cond), &(task->mutex))) {
         fprintf(stderr, "stdTriggerfun, pthread_cond_wait");
         pthread_mutex_unlock(&(task->mutex)); return;
      }
   }
   pthread_mutex_unlock(&(task->mutex));
   // printf("task %s running\n", task->name);
}

static void *doTask(void *p) {
   vm *task = (vm *)p;
   if (task->period > 0) {
      int32_t sec = task->period / 1000, nsec = (task->period % 1000) * 1000000;
      struct itimerspec tv = {sec, nsec, sec, nsec};

      task->triggerFun = stdTriggerFun;
      task->evp.sigev_notify = SIGEV_THREAD;
      task->evp.sigev_value.sival_ptr = p;
      task->evp.sigev_notify_function = wakeupPeriodicTask;
      if (timer_create(CLOCK_MONOTONIC, &(task->evp), &(task->tid))) {
         fprintf(stderr, "doTask, timer_create");
         return NULL;
      }
      if (timer_settime(task->tid, 0, &tv, NULL)) {
         fprintf(stderr, "doTask, timer_settime");
         return NULL;
      }
   } else {
      task->triggerInst->definition->armEvents(task);
   }
   // printf("triggerFun: %x\n", task->triggerFun);
   for(;;) {
      task->triggerFun(task);
      // printf("executing task... task = 0x%x\n", task);
      doFetchDecodeExecuteLoop(task);
      // printf("done with task\n");
   }
   return NULL;
}

char *projDir = "", *storeRoot = ""; 

static sigset_t signal_mask;


static void *signal_thread(void *arg) {
   int sig_caught;

   for (;;) {
      if (sigwait(&signal_mask, &sig_caught)) {
         fprintf(stderr, "signal_thread, cannot wait for signals");
         exit(EXIT_FAILURE);
      }
      fprintf(stderr, "Virtual machine: caught signal %d\n", sig_caught);
   }
}

int main(int argc, char **argv) {
   FILE *codeFile;
   vm InitVm;
   int32_t i, stkSiz, stk;
   int rc;
   pthread_t sig_thrd;

   printf("STAMPED runtime, V%d.%d\n", majorvers, minorvers);
   if (argc < 2) {
      printf("usage: automat instructionfile [--projectdir projectdir] [--logfile logfile] [--loglevel loglevel] [--storeroot storeroot]\n");
      exit(EXIT_SUCCESS); 
   }

   // mcheck(NULL);
   codeFile = fopen (argv[1], "rb");
   if (argc > 2) {
      int c = 2;
      for (;;) {
	 if (c + 1 >= argc) break;	
         if (strcmp(argv[c], "--projectdir") == 0) projDir = argv[c + 1];
         else if (strcmp(argv[c], "--storeroot") == 0) storeRoot = argv[c + 1];
         else if (strcmp(argv[c], "--logfile") == 0) {
            int log = open(argv[c + 1], O_WRONLY | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
            if (log < 0) {
               perror("main, cannot open log file");
               exit(EXIT_FAILURE);
            }
            if (dup2(log, 2) < 0) exit(EXIT_FAILURE);
            close(log);
         } else if (strcmp(argv[c], "--loglevel") == 0) loglevel = atoi(argv[c + 1]);
         c += 2;
      }
   }

   fprintf(stderr, "STAMPED runtime, V%d.%d\n", majorvers, minorvers);

   rc = pthread_attr_init(&vmAttrs);
   rc = pthread_attr_setstacksize(&vmAttrs, (0x20000 > PTHREAD_STACK_MIN) ? 0x20000 : PTHREAD_STACK_MIN);

   sigemptyset(&signal_mask);
   sigaddset(&signal_mask, SIGPIPE);
   if (pthread_sigmask(SIG_BLOCK, &signal_mask, NULL)) {
      fprintf(stderr, "main, cannot block signals");
      exit(EXIT_FAILURE);
   }   
   if (pthread_create(&sig_thrd, &vmAttrs, signal_thread, NULL)) {
      fprintf(stderr, "main, cannot create signal catcher");
      exit(EXIT_FAILURE);
   }

   InitVm = readBinary(codeFile);
   fclose(codeFile);

   rc = pthread_rwlock_init(&rw_global, NULL);
   if (rc) {
      fprintf(stderr, "main, pthread_rwlock_init");
      exit(EXIT_FAILURE);
   }
   initBlobMem();
   if (mlockall(MCL_CURRENT | MCL_FUTURE)) {
      perror("main, cannot lock memory");
      exit(EXIT_FAILURE);
   }
   printf("Running init code...\n");
   InitVm.running = 1;
   doFetchDecodeExecuteLoop(&InitVm);
   if (TskNo == 0) return 0;
   stkSiz = (constBegin - staticEnd - 2) / (TskNo + 1);
   stk = constBegin - 1 - stkSiz;
   for (i = 0; i < TskNo; i++) {
      Tasks[i].sp = Memory + stk; stk -= stkSiz;
      Tasks[i].running = 0;

      rc = pthread_cond_init(&(Tasks[i].cond), NULL); // Condition we are waiting for
      if (rc) {
         fprintf(stderr, "main, task cond's");
         exit(EXIT_FAILURE);
      }
      rc = pthread_mutex_init(&(Tasks[i].mutex), NULL); // Mutex for condition
      if (rc) {
         fprintf(stderr, "main task mutex's");
         exit(EXIT_FAILURE);
      }   
      if (i < TskNo - 1) {
         rc = pthread_create(&(Tasks[i].thread), &vmAttrs, doTask, (void *)(Tasks + i));
         if (rc) {
            fprintf(stderr, "main, pthread_create task threads");
            exit(EXIT_FAILURE);
         }
      }
   }
   if (TskNo > 0) doTask(Tasks + TskNo - 1);
   return 0;
}

