/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package poker;

/**
 *
 * @author Wolfgang Pree
 * @version     1.0
     */
public class HandRanker {
    /**
     * the core data structure for hand ranking is the following table representation of a hand:
     * rows: 4 suits
     * columns: 13 ranks from 2 to ace
     * each table cell (type: int) stores the number of cards of the particular rank and suit
     * 
     * in addition to the rows and columns described above we add one column that contains the row sums 
     * and one row that contains the column sums
     * 
     * We use the two-dimensional array
     *      int[][] handCalculator= new int[5][14];
     * for that purpose.
     * 
     * example: a hand consisting of the five cards 
     *              5-Heart
     *              Queen-Spade
     *              5-Heart
     *              10-Diamond
     *              10-Club
     * would thus be represented in handCalculator as follows, where the
     * - ranks are: 2, 3, 4, 5, 6, 7, 8, 8, T (for ten), J (for Jack), Q (for Queen), K (for King), A (for Ace)
     * - suits are: S (for Spade), H (for Heart), D (for Diamond), C (for Club)
     * <br><br>
     * __  2  3  4  5  6  7  8  9  T  J  Q  K  A   ∑        <br>
     * ----------------------------------------------       <br>
     * S | 0  0  0  0  0  0  0  0  0  0  1  0  0 | 1        <br>
     * H | 0  0  0  2  0  0  0  0  0  0  0  0  0 | 2        <br>
     * D | 0  0  0  0  0  0  0  0  1  0  0  0  0 | 1        <br>
     * C | 0  0  0  0  0  0  0  0  1  0  0  0  0 | 1        <br>
     * ----------------------------------------------       <br>
     * ∑ | 0  0  0  2  0  0  0  0  2  0  1  0  0 | -        <br><br>
     * 
     * 
     * This example illustrates that it is straight-forward to determine the hand value: The column sums show
     * two columns with value 2 representing the two pairs in the hand. A flush would require that the row sum of one suit
     * is 5 (see method isFlush()).
     * 
     * The table can also be used to sort a hand according to ranks by processing the columns 
     * from right (Ace-column) to left (2-column). This is required to compare two hands of same 
     * value (see method compareTo in class Hand). The hand in the example would be sorted according
     * to poker rules in method assessAndSort() as follows:
     * 
     *              10-Diamond
     *              10-Club
     *              5-Heart
     *              5-Heart
     *              Queen-Spade
     * 
     * Note that table cell handCalculator[13][4] is superfluous and not used. It always has 
     * the value 0 (not the '-' as shown in the sample table)
     * 
     * 
     * the core data structure for hand ranking is the following table representation of a hand:
     * rows: 4 suits
     * columns: 13 ranks from 2 to ace
     * each table cell (type: int) stores the number of cards of the particular rank and suit
     * 
     * in addition to the rows and columns described above we add one column that contains the row sums 
     * and one row that contains the column sums
     * 
     * We use the two-dimensional array
     *      int[][] handCalculator= new int[5][14];
     * for that purpose.
     * 
     * example: a hand consisting of the five cards 
     *              5-Heart
     *              Queen-Spade
     *              5-Heart
     *              10-Diamond
     *              10-Club
     * would thus be represented in handCalculator as follows, where the
     * - ranks are: 2, 3, 4, 5, 6, 7, 8, 8, T (for ten), J (for Jack), Q (for Queen), K (for King), A (for Ace)
     * - suits are: S (for Spade), H (for Heart), D (for Diamond), C (for Club)
     * <br><br>
     * __  2  3  4  5  6  7  8  9  T  J  Q  K  A   ∑    <br>
     * ----------------------------------------------   <br>
     * S | 0  0  0  0  0  0  0  0  0  0  1  0  0 | 1    <br>
     * H | 0  0  0  2  0  0  0  0  0  0  0  0  0 | 2    <br>
     * D | 0  0  0  0  0  0  0  0  1  0  0  0  0 | 1    <br>
     * C | 0  0  0  0  0  0  0  0  1  0  0  0  0 | 1    <br>
     * ----------------------------------------------   <br>
     * ∑ | 0  0  0  2  0  0  0  0  2  0  1  0  0 | -    <br><br>
     * 
     * 
     * This example illustrates that it is straight-forward to determine the hand value: The column sums show
     * two columns with value 2 representing the two pairs in the hand. A flush would require that the row sum of one suit
     * is 5 (see method isFlush()).
     * 
     * The table can also be used to sort a hand according to ranks by processing the columns 
     * from right (Ace-column) to left (2-column). This is required to compare two hands of same 
     * value (see method compareTo in class Hand). The hand in the example would be sorted according
     * to poker rules in method assessAndSort() as follows:
     * 
     *              10-Diamond
     *              10-Club
     *              5-Heart
     *              5-Heart
     *              Queen-Spade
     * 
     * Note that table cell handCalculator[13][4] is superfluous and not used. It always has 
     * the value 0 (not the '-' as shown in the sample table)
     * 
     * 
     */
    protected static int[][] handCalculator= new int[5][14];
    private static int currCard= 0;   // only used between calls of assessAndSort -> convertHandCalcColToCards
                                      //   what could be alternatives? are they better?
    private HandRanker() { // do not allow instantiation as HandRanker contains only static members
    }
    private static void reset() {
        for (int i= 0; i < 5; i++) {
            for (int j= 0; j < 14; j++) {
               handCalculator[i][j]= 0;
            }
        }
    }
    private static void setHand(Hand hand) {    // afterwards handCaluclator <=> hand with 
                                               //   properply calculated row and column sums
        reset();
        for (int i= 0; i < 5; i++) {
            handCalculator[hand.cards[i].suit.ordinal()][hand.cards[i].rank.ordinal()]++;
        }
        // calc row sums
        int rSum= 0;
        for (int s= 0; s < 4; s++) {
            for (int r= 0; r < 13; r++)
               rSum= rSum + handCalculator[s][r];
            handCalculator[s][13]= rSum;
            rSum= 0;
        }
        // calc column sums
        int cSum= 0;
        for (int r= 0; r < 13; r++) {
            for (int s= 0; s < 4; s++)
               cSum= cSum + handCalculator[s][r];
            handCalculator[4][r]= cSum;
            cSum= 0;
        }
        printHandCalc();
    }
    public static HandVal assessAndSort(Hand h) {
        setHand(h);
        HandVal v= assessHand();
        h.handVal= v;
        // sort hand based on rank and handVal according to official poker rules
        //  this means that the hand value (eg, one pair) overrides the rank: a hand
        //  with a Jack-pair and an Ace, King, and Queen would be sorted (in descending order) as
        //    Jack-Jack-Ace-King-Queen
        //  If that hand is a flush (all cards have same suit) then the descending order is
        //    Ace-King-Queen-Jack-Jack
        currCard= 0;
        // sort according to column sum; thus, the highest number of 'same of a kind' has priority over rank
        //   or as formulated above, the hand value (eg, one pair) overrides the rank
        for (int colSum= 5; colSum >=1; colSum--) {
            for (int r= 12; r >= 0; r--) {
                if (handCalculator[4][r] == colSum)
                    convertHandCalcColToCards(h, r);    // processes column r (ie, r-th rank) of handCalculator
            }
        }
        // treat special case 'straight with ace': ace needs to be valued as ONE
        if (v == HandVal.STRAIGHT && handCalculator[4][12] == 1) {
            Card tmp= h.cards[0];
            for (int s= 0; s < 4; s++)
                h.cards[s]= h.cards[s+1];
            h.cards[4]= tmp;
        } 
        // treat special case 'flush with pair/3 of a kind': sort only according to ranks and ignore pair/3 of a kind
        if (v == HandVal.FLUSH) {
            currCard= 0;
            for (int r= 12; r >= 0; r--)
                convertHandCalcColToCards(h, r);        // processes column r (ie, r-th rank) of handCalculator                   
        }
        return v;
    }
    private static void convertHandCalcColToCards(Hand h, int r) {  // for each rank r: r runs from 12 to 0
        for (int i= 0; i < 4; i++) { // for each suit s
            for (int k= 0; k < handCalculator[i][r]; k++) { // for each  card defined by particular rank+suit
                h.cards[currCard].rank= Rank.values()[r];
                h.cards[currCard].suit= Suit.values()[i];
                currCard++;
            }
        } 
    }
    private static HandVal assessHand() {  // private method => whether a hand was set need NOT to be checked!
        if (isStraight() && isFlush())
            return HandVal.STRAIGHTFLUSH;
        if (isPoker())
            return HandVal.POKER;
        if (isFullHouse())
            return HandVal.FULLHOUSE;
        if (isFlush())
            return HandVal.FLUSH;
        if (isStraight())
            return HandVal.STRAIGHT;
        if (isThreeOfAKind())
            return HandVal.THREEOFAKIND;
        if (isTwoPairs())
            return HandVal.TWOPAIRS;
        if (isOnePair())
            return HandVal.ONEPAIR;
        if (isHighCard())
            return HandVal.HIGHCARD;
        return HandVal.NOTVALID;
    }
    private static boolean isFlush() {
        for (int i= 0; i < 4; i++) {
            if (handCalculator[i][13] == 5)
                return true;
        }
        return false;
    }
    private static boolean isStraight() {
        String sumOfColsStr="";
        sumOfColsStr+= new Integer(handCalculator[4][12]).toString();  //no of aces as first digit in the string
        for (int j= 0; j < 13; j++)
            sumOfColsStr+= new Integer(handCalculator[4][j]).toString();
        return sumOfColsStr.contains("11111");
    }
    private static boolean isPoker() {
        for (int j= 0; j < 14; j++) {
            if (handCalculator[4][j] >= 4)
                return true;
        }
        return false;
    }
    private static boolean isFullHouse() {
        return isThreeOfAKind() && isOnePair();        
    }
    private static boolean isThreeOfAKind() {
        boolean threeFound= false;
        for (int j= 0; j < 14; j++) {
            if (handCalculator[4][j] == 3)
                threeFound= true;
        }
        return threeFound;        
    }
    private static boolean isTwoPairs() {
        return (noOfPairs() == 2);        
    }
    private static boolean isOnePair() {
        return (noOfPairs() == 1);        
    }
    private static int noOfPairs() {
        int noOfPairs= 0;
        for (int j= 0; j < 14; j++) {
            if (handCalculator[4][j] == 2)
                noOfPairs++;
        }
        return noOfPairs;               
    }
    private static boolean isHighCard() {
        boolean moreFound= false;
        for (int j= 0; j < 14; j++) {
            if (handCalculator[4][j] > 1)
                moreFound= true;
        }
        return !moreFound;        
    }
    private static void printHandCalc() {
        for (int i= 0; i < 5; i++) {
            for (int j= 0; j < 14; j++) {
               System.out.print(handCalculator[i][j]); System.out.print("  ");
            }
            System.out.println();
        }
    }
}