package IRed.spectro.stamped;

import java.util.List;
import java.util.LinkedList;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Set;
import java.util.Map;
import java.io.DataOutputStream;
import java.io.IOException;

public class Code {

   int StaticSize, cnstCnt;
   private int nxtLab = 0;

   public class Label {

      int no;
      int addr;
      boolean isSet;

      Label() {
         no = nxtLab;
         nxtLab++;
         isSet = false;
      }

   }

   public class Line {
      private int addr;
      private Instruction inst;
      private int AddrOrOffset;
      private Label target;
      private int size, count;
      private boolean local, lock, blob;
      private List<Integer> blobLst;
      private String comment;

      Line(int addr, Instruction inst, int AddrOrOffset, Label target, int size, int count, boolean local, boolean lock, boolean blob, List<Integer> blobLst, String comment) {
         this.addr = addr; this.inst = inst; this.AddrOrOffset = AddrOrOffset; this.target = target;
         this.size = size; this.count = count; this.local = local; this.lock = lock; this.blob = blob; 
         this.blobLst = blobLst; this.comment = comment;
      }

      public void dump() {
         if (inst.t1 == ArgTy.none) { 
            // -> ADD SUB MUL DIV MOD NEG AND OR XOR NOT LT LE EQ GE GT NEQ FADD FSUB FMUL FDIV FNEG FLT FLE FEQ FGE FGT FNEQ TOFLOAT TOINT NOP STOP
            System.out.println(addr + " "+ inst + "\t\t--" + comment);
         } else if (inst.t1 == ArgTy.blob && inst.t2 == ArgTy.none) { 
            // -> STOREV LOADV
            String blb = ((blob) ? " blob" : "");
            System.out.println(addr + " "+ inst + blb + "\t\t--" + comment);
         } else if ((inst.t1 == ArgTy.address || inst.t1 == ArgTy.offset) && inst.t2 == ArgTy.none) {
            // -> ADDFP JMP ANDJMP ORJMP FALSEJMP TRUEJMP CALL UPFORTST DWNFORTST FOREND TRY TRYEND THROW
            if (target != null) {
               System.out.println(addr + " " + inst + " " + target.addr + "\t\t--" + comment);
            } else 
               System.out.println(addr + " " + inst + " " + AddrOrOffset + "\t\t--" + comment);
         } else if ((inst.t1 == ArgTy.address || inst.t1 == ArgTy.offset) && inst.t2 == ArgTy.blob && inst.t3 == ArgTy.none) {
            // -> LOAD LOADL STORE STOREL
            String blb = ((blob) ? " blob" : "");
            System.out.println(addr + " " + inst + " " + AddrOrOffset + blb + "\t\t--" + comment);
         } else if (inst.t1 == ArgTy.size && inst.t2 == ArgTy.none) {
            // -> CHECK
            System.out.println(addr + " " + inst + " " + size + "\t\t--" + comment);
         } else if (inst.t1 == ArgTy.count && inst.t2 == ArgTy.none) {
            // -> RETURN COMPCALL EARLIEST LATEST
            System.out.println(addr + " " + inst + " " + count + "\t\t--" + comment);
         } else if (inst.t1 == ArgTy.size && inst.t2 == ArgTy.bloblst && inst.t3 == ArgTy.none) {
            // -> ENTER
            System.out.println(addr + " " + inst + " " + size + "\t\t--" + comment);
            if (blobLst != null) for (int o: blobLst) System.out.println("      " + o);
         } else if (inst.t1 == ArgTy.count && inst.t2 == ArgTy.bloblst && inst.t3 == ArgTy.none) {
            // -> LOADMV STOREMV
            System.out.println(addr + " " + inst + " " + count + "\t\t--" + comment);
            if (blobLst != null) for (int o: blobLst) System.out.println("      " + o);
         } else if ((inst.t1 == ArgTy.address || inst.t1 == ArgTy.offset) && inst.t2 == ArgTy.size && inst.t3 == ArgTy.blob && inst.t4 == ArgTy.none) {
            // -> LOADA LOADAL STOREA STOREAL
            String blb = ((blob) ? " blob" : "");
            System.out.println(addr + " " + inst + " " + AddrOrOffset + " " + size + blb + "\t\t--" + comment);
         } else if (inst.t1 == ArgTy.address && inst.t2 == ArgTy.local && inst.t3 == ArgTy.count && inst.t4 == ArgTy.lock && inst.t5 == ArgTy.bloblst) {
            // -> LOADM STOREM
            String lck = ((lock) ? " locked" : ""), lcl = ((local) ? " local" : ""); 
            System.out.println(addr + " " + inst + " " + AddrOrOffset + " : " + count + lck + lcl + "\t\t--" + comment);
            if (blobLst != null) for (int o: blobLst) System.out.println("      " + o);
         } else if (inst.t1 == ArgTy.count && inst.t2 == ArgTy.offset && inst.t3 == ArgTy.size && inst.t4 == ArgTy.bloblst && inst.t5 == ArgTy.none) {
            // -> XTRACT
            System.out.println(addr + " " + inst + " " + count + " " + AddrOrOffset + " " + size + "\t\t--" + comment);
            if (blobLst != null) for (int o: blobLst) System.out.println("      " + o);
         } else if (inst.t1 == ArgTy.count && inst.t2 == ArgTy.size && inst.t3 == ArgTy.none) {
            // -> FRETURN
            System.out.println(addr + " " + inst + " " + count + " " + size + "\t\t--" + comment);
         } else if (inst.t1 == ArgTy.count && inst.t2 == ArgTy.size && inst.t3 == ArgTy.bloblst && inst.t4 == ArgTy.none) {
            // -> STKAREF
            System.out.println(addr + " " + inst + " " + count + " " + size + "\t\t--" + comment);
            if (blobLst != null) for (int o: blobLst) System.out.println("      " + o);            
         } else if (inst.t1 == ArgTy.size && inst.t2 == ArgTy.count && inst.t3 == ArgTy.none) {
            // -> STOREP
            System.out.println(addr + " " + inst + " " + size + " " + count + "\t\t--" + comment);
         }
      }          

      void dumpBinary(DataOutputStream out) throws IOException {
         if (inst.t1 == ArgTy.none) { 
            // -> ADD SUB MUL DIV MOD NEG AND OR XOR NOT LT LE EQ GE GT NEQ FADD FSUB FMUL FDIV FNEG FLT FLE FEQ FGE FGT FNEQ TOFLOAT TOINT NOP STOP DURATION
            if (inst != Instruction.NOP) out.writeInt(inst.opCode << 24);
         } else if (inst.t1 == ArgTy.blob && inst.t2 == ArgTy.none) {    
            // -> STOREV LOADV
            out.writeInt((inst.opCode << 24) | ((blob) ? 0x800000 : 0x0));
         } else if ((inst.t1 == ArgTy.address || inst.t1 == ArgTy.offset) && inst.t2 == ArgTy.none) {
            // -> ADDFP JMP ANDJMP ORJMP FALSEJMP TRUEJMP CALL UPFORTST DWNFORTST FOREND TRY TRYEND THROW
            if (target != null) {
               if (!target.isSet) throw new FatalError("Label L#" + target.no + " is not set");
               out.writeInt((inst.opCode << 24) | ((target.addr - addr) & 0xFFFFF));
            } else 
               out.writeInt((inst.opCode << 24) | (AddrOrOffset & 0xFFFFF));
         } else if ((inst.t1 == ArgTy.address || inst.t1 == ArgTy.offset) && inst.t2 == ArgTy.blob && inst.t3 == ArgTy.none) {
            // -> LOAD LOADL STORE STOREL
            out.writeInt((inst.opCode << 24) | ((blob) ? 0x800000 : 0x0) | (AddrOrOffset & 0xFFFFF)); 
         } else if (inst.t1 == ArgTy.size && inst.t2 == ArgTy.none) {
            // -> CHECK
            out.writeInt((inst.opCode << 24) | (size & 0xFFFF));
         } else if (inst.t1 == ArgTy.count && inst.t2 == ArgTy.none) {
            // -> RETURN COMPCALL EARLIEST LATEST
            out.writeInt((inst.opCode << 24) | (count & 0xFFFF));
         } else if (inst.t1 == ArgTy.size && inst.t2 == ArgTy.bloblst && inst.t3 == ArgTy.none) {
            // -> ENTER
            int blobCnt = ((blobLst == null) ? 0 : blobLst.size());
            if (blobCnt > 63) throw new FatalError("too many BLOBs");
            out.writeInt((inst.opCode << 24) | ((blobCnt << 16) & 0x3F0000) | (size & 0xFFFF));
            if (blobLst != null) for (int o: blobLst) out.writeInt(o);   
         } else if (inst.t1 == ArgTy.count && inst.t2 == ArgTy.bloblst && inst.t3 == ArgTy.none) {
            // -> LOADMV STOREMV
            int blobCnt = ((blobLst == null) ? 0 : blobLst.size());
            if (blobCnt > 63) throw new FatalError("too many BLOBs");
            out.writeInt((inst.opCode << 24) | ((blobCnt << 16) & 0x3F0000) | (count & 0xFFFF));
            if (blobLst != null) for (int o: blobLst) out.writeInt(o);   
         } else if ((inst.t1 == ArgTy.address || inst.t1 == ArgTy.offset) && inst.t2 == ArgTy.size && inst.t3 == ArgTy.blob && inst.t4 == ArgTy.none) {
            // -> LOADA LOADAL STOREA STOREAL
            out.writeInt((inst.opCode << 24) | ((blob) ? 0x800000 : 0x0) | (AddrOrOffset & 0xFFFFF));
            out.writeInt(size & 0xFFFF);
         } else if (inst.t1 == ArgTy.address && inst.t2 == ArgTy.local && inst.t3 == ArgTy.count && inst.t4 == ArgTy.lock && inst.t5 == ArgTy.bloblst) {
            // -> LOADM STOREM
            out.writeInt((inst.opCode << 24) | ((local) ? 0x800000 : 0x0) | ((lock) ? 0x400000 : 0x0) | (AddrOrOffset & 0xFFFFF));
            int blobCnt = ((blobLst == null) ? 0 : blobLst.size());
            if (blobCnt > 63) throw new FatalError("too many BLOBs");
            out.writeInt(((blobCnt << 16) & 0x3F0000) | (count & 0xFFFF));
            if (blobLst != null) for (int o: blobLst) out.writeInt(o);
         } else if (inst.t1 == ArgTy.count && inst.t2 == ArgTy.offset && inst.t3 == ArgTy.size && inst.t4 == ArgTy.bloblst && inst.t5 == ArgTy.none) {
            // -> XTRACT
            int blobCnt = ((blobLst == null) ? 0 : blobLst.size());
            if (blobCnt > 63) throw new FatalError("too many BLOBs");
            out.writeInt((inst.opCode << 24) | ((blobCnt << 16) & 0x3F0000) | (AddrOrOffset & 0xFFFF));
            out.writeInt((count << 16) | (size & 0xFFFF));
            if (blobLst != null) for (int o: blobLst) out.writeInt(o);
         } else if (inst.t1 == ArgTy.count && inst.t2 == ArgTy.size && inst.t3 == ArgTy.none) {
            // -> FRETURN
               out.writeInt((inst.opCode << 24) | (count & 0xFFFF));
               out.writeInt(size & 0xFFFF);
         } else if (inst.t1 == ArgTy.count && inst.t2 == ArgTy.size && inst.t3 == ArgTy.bloblst && inst.t4 == ArgTy.none) {
            // -> STKAREF
            int blobCnt = ((blobLst == null) ? 0 : blobLst.size());
            if (blobCnt > 63) throw new FatalError("too many BLOBs");
            out.writeInt((inst.opCode << 24) | ((blobCnt << 16) & 0x3F0000));
            out.writeInt((count << 16) | (size & 0xFFFF));
            if (blobLst != null) for (int o: blobLst) out.writeInt(o);            
         } else if (inst.t1 == ArgTy.size && inst.t2 == ArgTy.count && inst.t3 == ArgTy.none) {
            // -> STOREP
            out.writeInt((inst.opCode << 24) | ((size << 14) & 0xFFC000) | (count & 0x3FFF));
         }
      }

   }

   private int PC;
   private ArrayList<Line> CodeMem;
   private HashMap<Integer, Integer> IntConsts;
   private HashMap<Float, Integer> FloatConsts;
   private HashMap<String, Integer> StringConsts;
   private HashMap<List<Integer>, Integer> BlobLstConsts;

   public Code() {
      StaticSize = 0;
      PC = 0;
      CodeMem = new ArrayList<Line>(); 
      cnstCnt = 0xFFFFF;
      IntConsts = new HashMap<Integer, Integer>();
      FloatConsts = new HashMap<Float, Integer>();
      StringConsts = new HashMap<String, Integer>();
      BlobLstConsts = new HashMap<List<Integer>, Integer>();
   }

   public void Emit(Instruction inst, String comment) {
      // ADD SUB MUL DIV MOD NEG AND OR XOR NOT LT LE EQ GE GT NEQ FADD FSUB FMUL FDIV FNEG FLT FLE FEQ FGE FGT FNEQ TOFLOAT TOINT NOP DURATION
      if (inst.t1 != ArgTy.none) throw new FatalError("Invalid instruction format");
      CodeMem.add(new Line(PC, inst, 0, null, 0, 0, false, false, false, null, comment));
      PC += inst.wordCount;
   }

   public void Emit(Instruction inst, boolean blob, String comment) {
      // STOREV LOADV
      if (inst.t1 != ArgTy.blob || inst.t2 != ArgTy.none)  
         throw new FatalError("Invalid instruction format " + inst);
      CodeMem.add(new Line(PC, inst, 0, null, 0, 0, false, false, blob, null, comment));
      PC += inst.wordCount;
   }

   public void Emit(Instruction inst, int arg, String comment) {
      if (inst.t2 != ArgTy.none)  throw new FatalError("Invalid instruction format " + inst);
      if (inst.t1 == ArgTy.size) 
         // CHECK
         CodeMem.add(new Line(PC, inst, 0, null, arg, 0, false, false, false, null, comment));
      else if (inst.t1 == ArgTy.count)
         // RETURN COMPCALL EARLIEST LATEST
         CodeMem.add(new Line(PC, inst, 0, null, 0, arg, false, false, false, null, comment));
      else if (inst.t1 == ArgTy.address || inst.t1 == ArgTy.offset)
         // ADDFP THROW
         CodeMem.add(new Line(PC, inst, arg, null, 0, 0, false, false, false, null, comment));
      else
         throw new FatalError("Invalid instruction format " + inst);
      PC += inst.wordCount;  
   }

   public void Emit(Instruction inst, int AddrOrOffset, boolean blob, String comment) {
      // LOAD LOADL STORE STOREL
      if ((inst.t1 != ArgTy.address && inst.t1 != ArgTy.offset) || inst.t2 != ArgTy.blob || inst.t3 != ArgTy.none)  
         throw new FatalError("Invalid instruction format " + inst);
      CodeMem.add(new Line(PC, inst, AddrOrOffset, null, 0, 0, false, false, blob, null, comment));
      PC += inst.wordCount;
   }


   public void Emit(Instruction inst, int arg, List<Integer> blobLst, String comment) {
      if (inst.t1 == ArgTy.count && inst.t2 == ArgTy.bloblst && inst.t3 == ArgTy.none)
         // LOADMV STOREMV
         CodeMem.add(new Line(PC, inst, 0, null, 0, arg, false, false, false, blobLst, comment));
      else if (inst.t1 == ArgTy.size && inst.t2 == ArgTy.bloblst && inst.t3 == ArgTy.none)
         // ENTER
         CodeMem.add(new Line(PC, inst, 0, null, arg, 0, false, false, false, blobLst, comment));      
      else
         throw new FatalError("Invalid instruction format " + inst);
      PC += inst.wordCount + ((blobLst != null) ? blobLst.size() : 0);  
   }

   public void Emit(Instruction inst, int AddrOrOffset, int size, boolean blob, String comment) {
      // LOADA LOADAL STOREA STOREAL
      if ((inst.t1 != ArgTy.address && inst.t1 != ArgTy.offset) || inst.t2 != ArgTy.size || inst.t3 != ArgTy.blob || inst.t4 != ArgTy.none)  
         throw new FatalError("Invalid instruction format " + inst);
      CodeMem.add(new Line(PC, inst, AddrOrOffset, null, size, 0, false, false, blob, null, comment));
      PC += inst.wordCount;
   }

   public void Emit(Instruction inst, int AddrOrOffset, int count, boolean local, boolean lock, List<Integer> blobLst, String comment) {
      // LOADM STOREM
      if ((inst.t1 != ArgTy.address && inst.t1 != ArgTy.offset) || inst.t2 != ArgTy.local || inst.t3 != ArgTy.count || inst.t4 != ArgTy.lock || inst.t5 != ArgTy.bloblst)  
         throw new FatalError("Invalid instruction format " + inst);
      CodeMem.add(new Line(PC, inst, AddrOrOffset, null, 0, count, local, lock, false, blobLst, comment));
      PC += inst.wordCount + ((blobLst != null) ? blobLst.size() : 0);
   }

   public void Emit(Instruction inst, Label target, String comment) {
      // JMP ANDJMP ORJMP FALSEJMP TRUEJMP CALL UPFORTST DWNFORTST FOREND TRY TRYEND
      if (inst.t1 != ArgTy.offset && inst.t2 != ArgTy.none) throw new FatalError("Invalid instruction format " + inst);
      CodeMem.add(new Line(PC, inst, 0, target, 0, 0, false, false, false, null, comment));
      PC += inst.wordCount;
   }

   public void Emit(Instruction inst, int RecSize, int Offset, int FldSize, List<Integer> blobLst, String comment) {
      // XTRACT
      if (inst.t1 != ArgTy.count || inst.t2 != ArgTy.offset || inst.t3 != ArgTy.size || inst.t4 != ArgTy.bloblst || inst.t5 != ArgTy.none) 
         throw new FatalError("Invalid instruction format " + inst);
      CodeMem.add(new Line(PC, inst, Offset, null, FldSize, RecSize, false, false, false, blobLst, comment));
      PC += inst.wordCount + ((blobLst != null) ? blobLst.size() : 0);  
   }

   public void Emit(Instruction inst, int baseSize, int fldSize, List<Integer> blobLst, String comment) {
      // STKAREF
      if (inst.t1 != ArgTy.count || inst.t2 != ArgTy.size || inst.t3 != ArgTy.bloblst || inst.t4 != ArgTy.none) 
         throw new FatalError("Invalid instruction format " + inst);
      CodeMem.add(new Line(PC, inst, 0, null, fldSize, baseSize, false, false, false, blobLst, comment));
      PC += inst.wordCount + ((blobLst != null) ? blobLst.size() : 0);  
   }

   public void Emit(Instruction inst, int leftOrBase, int rightOrFld, String comment) {
      // STOREP FRETURN
      if (inst.t1 == ArgTy.size && inst.t2 == ArgTy.count && inst.t3 == ArgTy.none) 
         CodeMem.add(new Line(PC, inst, 0, null, leftOrBase, rightOrFld, false, false, false, null, comment));
      else if (inst.t1 == ArgTy.count && inst.t2 == ArgTy.size && inst.t3 == ArgTy.none) 
         CodeMem.add(new Line(PC, inst, 0, null, rightOrFld, leftOrBase, false, false, false, null, comment));
      else          
         throw new FatalError("Invalid instruction format " + inst);
      PC += inst.wordCount;  
   }

   public int EmitConst(int val) {
      Integer addr = IntConsts.get(val);
      if (addr == null) {
         IntConsts.put(val, cnstCnt);
         addr = cnstCnt--;
      }    
      return addr;
   }

   public int EmitConst(float val) {
      Integer addr = FloatConsts.get(val);
      if (addr == null) {
         FloatConsts.put(val, cnstCnt);
         addr = cnstCnt--;
      }    
      return addr;
   }

   public int EmitConst(String s) {
      Integer addr = StringConsts.get(s);
      if (addr == null) {
         StringConsts.put(s, cnstCnt);
         addr = cnstCnt--;
      }
      return addr;
   }

   public int EmitConsts(List<Integer> l) {
      Integer addr = BlobLstConsts.get(l);
      if (addr == null) {
         cnstCnt -= l.size();
         BlobLstConsts.put(l, addr = cnstCnt + 1);
      }
      return addr;
   }

   public Label genLabel() {
      return new Label();
   }

   public void setLabel(Label l) {
      if (!l.isSet) {
         l.addr = PC;
         l.isSet = true;
      } else throw new FatalError("Label L#" + l.no + " is set twice");
   }

   public void dump() {
      for (Line l: CodeMem) l.dump();
      System.out.println();
   }

   public void dumpBinary(DataOutputStream out, SymbolTable sharedSyms, LinkedList<TaskAST> tasks) throws IOException {
      System.out.println("CodeSize: " + PC);
      out.writeInt(PC);
      for (Line l: CodeMem) l.dumpBinary(out);
      // Component Definitions
      sharedSyms.dumpCompDefsBinary(out);
      // Component Instances
      sharedSyms.dumpCompInstsBinary(out);
      // Tasks
      out.writeInt(tasks.size());
      for (TaskAST task: tasks) {
         System.out.println("writing task " + task.name);
         task.dumpBinary(out);
      }
      //  Bounds of space for stacks
      System.out.println("StaticSize: " + StaticSize);
      System.out.println("ConstBegin: " + cnstCnt);
      out.writeInt(StaticSize);
      out.writeInt(cnstCnt);
      // Integer constants
      Set<Map.Entry<Integer, Integer>> IntConstsSet = IntConsts.entrySet();
      out.writeInt(IntConstsSet.size());
      for (Map.Entry<Integer, Integer> e: IntConstsSet) {
         out.writeInt(e.getValue()); out.writeInt(e.getKey());    // Address, value
      }
      // Float constants
      Set<Map.Entry<Float, Integer>> FloatConstsSet = FloatConsts.entrySet();
      out.writeInt(FloatConstsSet.size());
      for (Map.Entry<Float, Integer> e: FloatConstsSet) {
         out.writeInt(e.getValue()); out.writeFloat(e.getKey());  // Address, value
      }
      // BlobList constants
      Set<Map.Entry<List<Integer>, Integer>> BlobLstConstsSet = BlobLstConsts.entrySet();
      out.writeInt(BlobLstConstsSet.size());
      for (Map.Entry<List<Integer>, Integer> e: BlobLstConstsSet) {
         out.writeInt(e.getValue());
         List<Integer> bl = e.getKey();
         out.writeInt(bl.size());
         for (Integer o: bl) out.writeInt(o);
      }
      // String constants
      Set<Map.Entry<String, Integer>> StringConstsSet = StringConsts.entrySet();
      out.writeInt(StringConstsSet.size());
      for (Map.Entry<String, Integer> e: StringConstsSet) {
         out.writeInt(e.getValue());    // Address, size chars
         String s = e.getKey();
         out.writeInt(s.length());
         out.writeBytes(s);  
      }
      int CompVarTotal = sharedSyms.CompVarCnt;
      for (TaskAST task: tasks) CompVarTotal += task.taskSyms.CompVarCnt;
      System.out.println("CompvarTotal = " + CompVarTotal);
      out.writeInt(CompVarTotal);
      sharedSyms.dumpCompVarsBinary(out);
      for (TaskAST task: tasks) task.taskSyms.dumpCompVarsBinary(out);
   } 

}
