package IRed.spectro.stamped;

import java.util.Map;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.ArrayList;
import java.io.DataOutputStream;
import java.io.IOException;

class SymbolTable {
   int size;
   int parSize;
   int startAddr;
   int maxOffset;
   int CompDefCnt;
   int CompInstCnt;
   int CompVarCnt;
   int EventCnt;
   LinkedHashMap<String, Symbol> table;
   
   // for implementing scopes
   SymbolTable outer;

   // for global scope, components and records
   public SymbolTable() {
      outer = null;
      size = 0;
      parSize = 0;
      maxOffset = 0;
      CompDefCnt = CompVarCnt = CompInstCnt = EventCnt = 0;
      table = new LinkedHashMap<String, Symbol>();
   }

   public SymbolTable(SymbolTable o) {   
      outer = o;
      startAddr = 0;
      size = 0;
      table = new LinkedHashMap<String, Symbol>();
   }

   public SymbolTable(SymbolTable o, int startAddr) {   
      outer = o;
      this.startAddr = startAddr;
      size = 0;
      table = new LinkedHashMap<String, Symbol>();
   }

   public Symbol find(String name) {
      SymbolTable scope = this;
      while (scope != null) {
         Symbol e = scope.table.get(name);
         if (e != null) {
            return e;
         }
         scope = scope.outer;
      }
      return null;
   }

   public boolean insert(String name, DataType type, boolean ro, Kind k) {
//      System.out.println("SymbolTable insert: name = " + name + " kind = " + k);
      if (table.containsKey(name)) return false;
      Symbol e = new Symbol(type, ro, k);
      table.put(name, e);
      if (k == Kind.COMPNAME) CompDefCnt++;
      if (k == Kind.COMPVAR) CompVarCnt++;
      if (type instanceof ComponentTy && k == Kind.SHAREDVAR) CompInstCnt++;
      if (type instanceof EventType) {
         e.offset = EventCnt;
         EventCnt += ((EventType)type).no;
      }
      return true;
   }

   public boolean insert(String name, DataType type, TypedAST a, boolean ro, Kind k) {
      if (table.containsKey(name)) return false;
      Symbol e = new Symbol(type, a, ro, k);
      table.put(name, e);
      if (k == Kind.COMPNAME) CompDefCnt++;
      if (k == Kind.COMPVAR) CompVarCnt++;
      if (type instanceof ComponentTy && k == Kind.SHAREDVAR) CompInstCnt++;
      if (type instanceof EventType) {
         e.offset = EventCnt;
         EventCnt += ((EventType)type).no;
      }
      return true;
   }

   public boolean insert(String name, TypedAST a, boolean ro, Kind k) {
//      System.out.println("insert ast " + name + a);
      if (table.containsKey(name)) return false;
      Symbol e = new Symbol(a, ro, k);
      table.put(name, e);
      if (k == Kind.COMPNAME) CompDefCnt++;
      if (k == Kind.COMPVAR) CompVarCnt++;
      if (a.type instanceof ComponentTy && k == Kind.SHAREDVAR) CompInstCnt++;
      if (a.type instanceof EventType) {
         e.offset = EventCnt;
         EventCnt += ((EventType)a.type).no;
      }
      return true;
   }

   public void copy(SymbolTable compTable) {
      // System.out.println("in SymbolTable copy");
      for (Map.Entry<String, Symbol> e: compTable.table.entrySet()) {
         Symbol s = e.getValue();
         table.put(e.getKey(), new Symbol(s.type, s.ast, s.readOnly, Kind.STATICVAR));
         // System.out.println("in SymbolTable copy: copying " + e.getKey());
      }
   }

   public boolean resolveConsts() {
      boolean ok = true;
      for (Symbol s: table.values()) {
         if (s.kind == Kind.CONSTNAME) {
            if (!s.resolved) s.ast = s.ast.resolveConst();
            if (!s.ast.isConst()) ok = false;
            s.type = s.ast.type;
            s.resolved = true;
         }
      }
      return ok;
   }

   public void dump(String left) {
      for (Map.Entry<String, Symbol> e: table.entrySet()) {
         Symbol s = e.getValue();
         System.out.println(left + e.getKey() + ": " + s.offset + ", " + s.kind);
         if (s.type != null) s.type.dump(left + "   ");
         if (s.ast != null) s.ast.dump(left + "   ");
      }
   }

   public boolean resolveTypes() {
      // System.out.println("resolving symboltable");
      for (Symbol s: table.values()) {
         if (s.kind == Kind.TYPENAME) {
            if (!s.resolved) {
               s.resolved = true;
               s.type = s.type.resolveType();
            }
         }
      }

      for (Symbol s: table.values()) {
         if (s.type instanceof UnknownType) return false;
      }

      return true;
   }

   public int allocateMemory(boolean isLocal, int parcnt) {
      return allocateMemory(isLocal, false, parcnt, startAddr);
   }

   public void allocateForVar() {
      // System.out.println("SymbolTable.allocateForVar, outer.maxOffset = " + outer.maxOffset);
      if (outer.maxOffset >= 0) allocateMemory(false, true, 0, -2);
      else allocateMemory(false, true, 0, outer.maxOffset - 2);
      // dump("ForVar ");
   }

   private int allocateMemory(boolean isLocal, boolean isFor, int parcnt, int offset) {
      int i;
      if (isLocal) {
         offset = 2;
         i = 0; 
         for (Symbol s: table.values()) {
            if (i == parcnt) break;
            int siz;
            if (s.kind == Kind.VARPAR) siz = 1;
            else if (s.kind == Kind.STATICVAR || s.kind == Kind.SHAREDVAR || s.kind == Kind.LOCALVAR || s.kind == Kind.COMPVAR) siz = s.type.size;
            else siz = 0;
            offset +=  siz;
            i++;
         }
         parSize = offset - 2;
      }
      i = 0; size = 0;
      for (Symbol s: table.values()) {
         s.mapped = true;
         if (isLocal) {
            if (i == parcnt) offset = -1;    // make space for pc, fp and pointer to ENTER (for Bloblist)
            i++;
         }
         int siz;
         if (s.kind == Kind.VARPAR) siz = 1;
         else if (s.kind == Kind.STATICVAR || s.kind == Kind.SHAREDVAR || s.kind == Kind.LOCALVAR || s.kind == Kind.COMPVAR || s.kind == Kind.FIELDNAME) {
            siz = s.type.size;
            if ((s.kind == Kind.STATICVAR || s.kind == Kind.SHAREDVAR) && s.type instanceof ComponentTy) siz += ((ComponentTy)s.type).syms.size;
         } else siz = 0;
         if (isLocal || isFor) offset -= siz;
         if (s.kind == Kind.COMPVAR && s.ast != null) {
            s.mapped = false; siz = 0;
         } else if (!(s.type instanceof EventType)) s.offset = offset;
         size += siz;
         if (!(isLocal || isFor)) offset += siz;
      }
      maxOffset = offset;
      // do the mapped variables...
      for (Symbol s: table.values()) {
         boolean ok = true;
         if (s.kind == Kind.COMPVAR && s.ast != null) {
            if (!(s.ast instanceof QualAST)) s.ast.SemErr("invalid MAP TO target");
            else {
               QualAST p = (QualAST)s.ast;
               if (p.distributeTypes(false)) {
                  if (!(p.base.type instanceof ComponentTy)) {p.SemErr("invalid MAP TO target"); ok = false;}
                  if (s.type instanceof ArrayType && p.qual instanceof ArrayRefAST) {
                     // map an array, to allow atomic access
                     if (!((ArrayType)s.type).basicTy.assignable(p.qual.type)) {p.SemErr("type missmatch in MAP TO"); ok = false;}
                     ArrayRefAST q = (ArrayRefAST)p.qual;
                     q.indexAst = q.indexAst.moveVal2Stack();
                     if (!(q.indexAst instanceof ConstIntAST)) {p.SemErr("invalid MAP TO: address not constant"); ok = false;}
                     int o = ((ConstIntAST)q.indexAst).val;
                     if (((ArrayType)s.type).no + o > ((ArrayType)q.arrayVarAst.type).no) {p.SemErr("invalid MAP TO: space exhausted"); ok = false;}  
                  } else if (!s.type.assignable(p.qual.type)) {p.SemErr("type missmatch in MAP TO"); ok = false;}
                  if (ok) {
                     p.base.moveAddr2Stack();
                     p.qual.moveAddr2Stack();
                     if (p.base.place != Place.CONSTADDR || p.qual.place != Place.CONSTADDR) p.SemErr("invalid MAP TO: address not constant");
                     else {
                        s.offset = p.base.offset + 1 + p.qual.offset;
                        s.readOnly = p.qual.readOnly;
                        s.mapped = true;
                     }
                  }
               }
            }
         }
      }
      if (isLocal && i == parcnt) offset = -1; 
      return maxOffset;
   }

   public List<Integer> getBlobOffsets(int strt) {
      // System.out.println("in BlobOffsets");
      List<Integer> Offs = null;
      for (Symbol s: table.values()) {
         if (s.kind != Kind.VARPAR) {
            List<Integer> eOffs = s.type.getBlobOffsets(strt + s.offset);
            if (eOffs != null) {
               if (Offs == null) Offs = new ArrayList<Integer>();
               Offs.addAll(eOffs);
            }
         }
      }
      // if (Offs != null) for (Integer o: Offs) System.out.println(o);
      // System.out.println("out BlobOffsets");
      return Offs;
   }

   public void generateCode() {
      for (Symbol s: table.values()) {
         if (s.kind == Kind.PROCNAME) {
            Compiler.code.Emit(Instruction.NOP, "Procedure");
            ProcType proc = ((ProcType)s.type);
            proc.locals.allocateMemory(true, proc.parcnt);
            // proc.locals.dump("");
            if (proc.body.distributeTypes()) {
               // proc.body.dump("");
               Compiler.code.setLabel(proc.label);
               Compiler.code.Emit(Instruction.ENTER, ((proc.locals.maxOffset < 0) ? -proc.locals.maxOffset - 1 : 0), proc.locals.getBlobOffsets(0), "");
               proc.body.generateCode();
               if (proc.retType == null) {
                  Compiler.code.Emit(Instruction.RETURN, proc.locals.parSize, "");
               }
            }
         }
      }
   }

   public void dumpCompDefsBinary(DataOutputStream out) throws IOException {
      out.writeInt(CompDefCnt);
      for (Map.Entry<String, Symbol> e: table.entrySet()) {
         if (e.getValue().kind == Kind.COMPNAME) {
            String name = e.getKey();
            out.writeInt(name.length()); out.writeBytes(name);
            ((ComponentTy)e.getValue().type).syms.dumpCompProcsBinary(out);
         }
      }
   }

   public void dumpCompProcsBinary(DataOutputStream out) throws IOException {
      int no = 0;
      for (Symbol s: table.values()) if (s.kind == Kind.PROCNAME) no++;
      out.writeInt(no);
      for (Symbol s: table.values()) {
         if (s.kind == Kind.PROCNAME) {
            s.type.dumpBinary(out);
         }
      }
   }

   public void dumpCompInstsBinary(DataOutputStream out) throws IOException {
      // System.out.println("dumpCompInstsBinary, CompInstCnt = " + CompInstCnt);
      out.writeInt(CompInstCnt);
      for (Map.Entry<String, Symbol> e: table.entrySet()) {
         Symbol s = e.getValue();
         if (s.type instanceof ComponentTy && s.kind == Kind.SHAREDVAR) {
            String instName = e.getKey();
            String compName = ((ComponentTy)s.type).name;
            out.writeInt(s.offset);
            out.writeInt(instName.length()); out.writeBytes(instName);
            out.writeInt(compName.length()); out.writeBytes(compName);
            out.writeInt(((ComponentTy)s.type).syms.EventCnt);
         }
      }
   }

   public void dumpCompVarsBinary(DataOutputStream out) throws IOException {
      for (Map.Entry<String, Symbol> e: table.entrySet()) {
         Symbol s = e.getValue();
         if (s.kind == Kind.COMPVAR) {
            String varName = e.getKey();
            System.out.println("dumpCompVarsBinary: name = " + varName);
            out.writeInt(varName.length()); out.writeBytes(varName);
            s.type.dumpBinary(out);
            out.writeInt(s.offset); out.writeInt(s.offset + s.type.size - 1);
            // System.out.println("dumping " + varName + " address = " + s.offset + " last address = " + (s.offset + s.type.size - 1));
         }
      }
   }

}
